package ai.grakn.kgms.console;

import ai.grakn.GraknConfigKey;
import ai.grakn.engine.GraknConfig;
import ai.grakn.kgms.console.command.CommandBus;
import ai.grakn.kgms.console.command.PrintWriterNotRecognised;
import ai.grakn.kgms.console.command.createuser.CreateUserCommand;
import ai.grakn.kgms.console.command.createuser.CreateUserDisplay;
import ai.grakn.kgms.console.command.deleteuser.DeleteUserCommand;
import ai.grakn.kgms.console.command.deleteuser.DeleteUserDisplay;
import ai.grakn.kgms.console.command.retrieveallusers.RetrieveAllUserCommand;
import ai.grakn.kgms.console.command.retrieveallusers.RetrieveAllUserDisplay;
import ai.grakn.kgms.console.command.retrieveuser.RetrieveUserCommand;
import ai.grakn.kgms.console.command.retrieveuser.RetrieveUserDisplay;
import ai.grakn.kgms.console.command.updateuser.UpdateUserCommand;
import ai.grakn.kgms.console.command.updateuser.UpdateUserDisplay;
import ai.grakn.kgms.console.rpc.GrpcChannelFactory;
import ai.grakn.kgms.console.rpc.ResponseListener;
import ai.grakn.kgms.console.rpc.UserManagementClient;
import ai.grakn.kgms.rpc.generated.KGMSConsoleGrpc;
import ai.grakn.util.SimpleURI;
import io.grpc.ManagedChannel;
import jline.console.ConsoleReader;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.nio.file.Paths;
import java.util.concurrent.ArrayBlockingQueue;

/**
 * KGMS Authenticated Console that can be run from command line
 *
 * @author Marco Scoppetta
 */


public class KGMSConsole {

    private static final Logger LOG = LoggerFactory.getLogger(KGMSConsole.class);

    private static final String PROMPT = ">>> ";
    private static final String USERNAME = "u";
    private static final String PASSWORD = "p";
    private static final String CONTACT_POINT = "c";
    private static final String TLS = "tls";
    private static final String TRUST_MANAGER = "tm";
    private static final String EXIT_COMMAND = "exit";

    private PrintWriter outPrinter;

    private final ConsoleReader console;
    private final ManagedChannel channel;
    private final CommandBus commandBus;
    private final UserManagementClient userManagementClient;

    public static void main(String[] args) {
        int exitCode = (startConsole(args, System.in, System.out, GraknConfig.create())) ? 0 : 1;
        System.exit(exitCode);
    }


    public static boolean startConsole(String[] args, InputStream in, OutputStream out, GraknConfig config) {
        try {
            CommandLine cmd = initialiseCommandLine(args);
            String username = cmd.getOptionValue('u');
            String password = cmd.getOptionValue('p');
            ManagedChannel channel = getChannel(cmd, config);

            KGMSConsole console = new KGMSConsole(in, out, channel);
            console.start(username, password);
        } catch (IOException | ParseException e) {
            LOG.error("Unable to start KGMS console", e);
            return false;
        }

        return true;
    }


    KGMSConsole(InputStream in, OutputStream out, ManagedChannel channel) throws IOException {
        this.console = new ConsoleReader(in, out);
        this.outPrinter = new PrintWriter(console.getOutput());
        this.channel = channel;
        ResponseListener responseListener = new ResponseListener(new ArrayBlockingQueue<>(1));
        this.userManagementClient = new UserManagementClient(KGMSConsoleGrpc.newStub(channel), responseListener);
        this.commandBus = initialiseCommandBus();
        registerShutdownHook();
    }

    void start(@Nullable String username, @Nullable String password) throws IOException {
        if (username == null) {
            username = console.readLine("Username: ");
        }
        if (password == null) {
            password = console.readLine("Passsword: ", '*');
        }
        userManagementClient.login(username, password);

        authenticatedShell();
    }

    private void authenticatedShell() throws IOException {
        console.setPrompt(PROMPT);
        String line;
        while ((line = console.readLine()) != null) {
            // Handle exit command here as we need to close client and channel
            if (line.trim().equals(EXIT_COMMAND)) {
                close();
                return;
            }
            commandBus.run(line);
        }
    }

    void close() {
        userManagementClient.close();
        this.channel.shutdown();
    }


    // --------  Helpers  ---------- //

    private CommandBus initialiseCommandBus() {
        return new CommandBus(new PrintWriterNotRecognised(this.outPrinter),
                new CreateUserCommand(userManagementClient, new CreateUserDisplay(this.outPrinter)),
                new RetrieveAllUserCommand(userManagementClient, new RetrieveAllUserDisplay(this.outPrinter)),
                new DeleteUserCommand(userManagementClient, new DeleteUserDisplay(this.outPrinter)),
                new RetrieveUserCommand(userManagementClient, new RetrieveUserDisplay(this.outPrinter)),
                new UpdateUserCommand(userManagementClient, new UpdateUserDisplay(this.outPrinter)));
    }


    private static ManagedChannel getChannel(CommandLine cmd, GraknConfig config) {
        // Define contact point with Server
        SimpleURI contactPoint;
        if (cmd.hasOption('c')) {
            contactPoint = new SimpleURI(cmd.getOptionValue('c'));
        } else {
            contactPoint = new SimpleURI(config.getProperty(GraknConfigKey.SERVER_HOST_NAME) + ":" + config.getProperty(GraknConfigKey.GRPC_PORT));
        }
        // Determine whether the connection needs to be encrypted
        boolean encryptionEnabled = Boolean.valueOf(cmd.getOptionValue(TLS));
        // Load path to trust manager for self-signed certificates (this is optional)
        String trustManager = cmd.getOptionValue(TRUST_MANAGER);
        if (encryptionEnabled) {
            if (trustManager == null) {
                return GrpcChannelFactory.getEncryptedChannel(contactPoint);
            } else {
                return GrpcChannelFactory.getEncryptedChannel(contactPoint, Paths.get(trustManager));
            }
        } else {
            return GrpcChannelFactory.getChannel(contactPoint);
        }
    }

    private static CommandLine initialiseCommandLine(String[] args) throws ParseException {
        Options options = new Options();

        //TODO: unify this options object with KGMSShellOptions
        options.addOption(USERNAME, "username", true, "username of admin user");
        options.addOption(PASSWORD, "password", true, "password of admin user");
        options.addOption(CONTACT_POINT, "contact-point", true, "contact point for gRPC server");
        options.addOption(TLS, "enable-tls", false, "enable TLS encryption");
        options.addOption(TRUST_MANAGER, "trust-manager", true, "path to trust manager for self-signed certificates");

        CommandLineParser parser = new DefaultParser();
        CommandLine cmd;
        cmd = parser.parse(options, args);
        return cmd;
    }

    /**
     * If user uses CTRL+C to close console we try our best to terminate gracefully.
     */
    private void registerShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> close()));
    }

}