package ai.grakn.kgms.console.rpc;


import ai.grakn.util.SimpleURI;
import io.grpc.ManagedChannel;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.NettyChannelBuilder;

import javax.net.ssl.SSLException;
import java.io.File;
import java.nio.file.Path;

/**
 *
 * Factory used to provide gRPC channels.
 *
 *  - plain text channel, no encryption.
 *  - encrypted channel over TLS with valid/public Certificate from server
 *  - encrypted channel over TLS with self-signed Certificate from server
 *
 * @author Marco Scoppetta
 */

public class GrpcChannelFactory {
    public static final int GRPC_MAX_MESSAGE_SIZE_IN_BYTES = Integer.MAX_VALUE; // 2 GB

    // Get channel that works over plain text
    public static ManagedChannel getChannel(SimpleURI uri){
        return NettyChannelBuilder
                .forAddress(uri.getHost(), uri.getPort())
                .maxInboundMessageSize(GRPC_MAX_MESSAGE_SIZE_IN_BYTES)
                .usePlaintext(true)
                .build();
    }

    // Get encrypted channel over TLS using self-signed Certificate
    public static ManagedChannel getEncryptedChannel(SimpleURI uri, Path trustedCertPath){
        File trustedCert = new File(trustedCertPath.toString());

        try {
            return NettyChannelBuilder
                    .forAddress(uri.getHost(), uri.getPort())
                    .maxInboundMessageSize(GRPC_MAX_MESSAGE_SIZE_IN_BYTES)
                    .sslContext(GrpcSslContexts.forClient().trustManager(trustedCert).build())
                    .build();

        } catch (SSLException e) {
            throw new RuntimeException(e);
        }
    }

    // Get encrypted channel over TLS using public Certificate
    public static ManagedChannel getEncryptedChannel(SimpleURI uri){
        return NettyChannelBuilder.forAddress(uri.getHost(), uri.getPort())
                .maxInboundMessageSize(GRPC_MAX_MESSAGE_SIZE_IN_BYTES)
                .build();
    }
}
