package ai.grakn.kgms.console.rpc;

import ai.grakn.kgms.authentication.model.User;
import ai.grakn.kgms.rpc.generated.KGMSConsoleGrpc;
import ai.grakn.kgms.rpc.generated.KGMSConsoleProto;
import io.grpc.stub.StreamObserver;

import java.util.List;
import java.util.stream.Collectors;

/**
 * Class that provides use cases for User management over gRPC
 */

public class UserManagementClient {

    private StreamObserver<KGMSConsoleProto.UserManagement.Req> requestSender;
    private final ResponseListener responseListener;
    private boolean terminated = false;

    public UserManagementClient(KGMSConsoleGrpc.KGMSConsoleStub stub, ResponseListener responseListener) {
        this.responseListener = responseListener;
        requestSender = stub.userManagement(responseListener);
    }

    public void login(String username, String password) {
        KGMSConsoleProto.Login.Req userRequest = KGMSConsoleProto.Login.Req.newBuilder()
                .setUsername(username)
                .setPassword(password)
                .build();
        KGMSConsoleProto.UserManagement.Req loginReq = KGMSConsoleProto.UserManagement.Req.newBuilder().setLoginReq(userRequest).build();
        send(loginReq);
    }

    /**
     * Send create user request to server
     * NOTE: we ignore the response as we don't want to show it to the user
     */
    public void create(String username, String password, String role) {
        KGMSConsoleProto.Create.Req createReq = KGMSConsoleProto.Create.Req.newBuilder()
                .setUsername(username)
                .setPassword(password)
                .setRole(role)
                .build();
        KGMSConsoleProto.UserManagement.Req req = KGMSConsoleProto.UserManagement.Req.newBuilder()
                .setCreateReq(createReq).build();
        send(req);
    }

    /**
     * Send delete user request to server
     * NOTE: we ignore the response as we don't want to show it to the user
     */
    public void delete(String username) {
        KGMSConsoleProto.Delete.Req deleteReq = KGMSConsoleProto.Delete.Req.newBuilder()
                .setUsername(username).build();
        KGMSConsoleProto.UserManagement.Req req = KGMSConsoleProto.UserManagement.Req.newBuilder()
                .setDeleteReq(deleteReq).build();
        send(req);
    }

    /**
     * Send retrieve user request to server and return new User
     */
    public User retrieve(String username) {
        KGMSConsoleProto.Retrieve.Req retrieveReq = KGMSConsoleProto.Retrieve.Req.newBuilder()
                .setUsername(username).build();
        KGMSConsoleProto.UserManagement.Req req = KGMSConsoleProto.UserManagement.Req.newBuilder()
                .setRetrieveReq(retrieveReq).build();
        KGMSConsoleProto.UserManagement.Res response = send(req);
        return new User(response.getRetrieveRes().getUser().getUsername(), response.getRetrieveRes().getUser().getRole());
    }

    /**
     * Send retrieve all users request to server and return new User
     */
    public List<User> retrieveAll() {
        KGMSConsoleProto.RetrieveAll.Req retrieveAllReq = KGMSConsoleProto.RetrieveAll.Req.newBuilder().build();
        KGMSConsoleProto.UserManagement.Req req = KGMSConsoleProto.UserManagement.Req.newBuilder()
                .setRetrieveAllReq(retrieveAllReq).build();
        KGMSConsoleProto.UserManagement.Res response = send(req);
        return response.getRetrieveAllRes().getUsersList()
                .stream()
                .map(user -> new User(user.getUsername(), user.getRole()))
                .collect(Collectors.toList());
    }

    /**
     * Send create user request to server
     * NOTE: we ignore the response as we don't want to show it to the user
     */
    public void update(String username, String password, String role) {
        KGMSConsoleProto.Update.Req.Builder updateReqBuilder = KGMSConsoleProto.Update.Req.newBuilder()
                .setUsername(username);
        if (password != null) updateReqBuilder.setPassword(password);
        if (role != null) updateReqBuilder.setRole(role);
        KGMSConsoleProto.UserManagement.Req req = KGMSConsoleProto.UserManagement.Req.newBuilder()
                .setUpdateReq(updateReqBuilder.build()).build();
        send(req);
    }

    /**
     * Send UserManagement Request to Server and wait for response
     *
     * @return UserManagement Response
     */
    private KGMSConsoleProto.UserManagement.Res send(KGMSConsoleProto.UserManagement.Req req) {
        if (terminated) throw new RuntimeException("Connection is already closed.");
        try {
            requestSender.onNext(req);
            GrpcResponse response = responseListener.getResponse();

            if (response.isException()) throw response.getException();
            if (response.isCompleted()) close(); //Note: this should never be the case, as it's always the Client completing the communication.

            return response.getResponse();
        } catch (Throwable e) {
            close();
            // Re throw Runtime exception as the client should terminate when getting exception from the Server
            throw new RuntimeException(e);
        }
    }

    public void close() {
        if (terminated) return;
        requestSender.onCompleted();
        terminated = true;
    }
}
