package deepwater.datasets;

// Inspired from http://stackoverflow.com/questions/8286668/how-to-read-mnist-data-in-c

import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.*;
import java.util.zip.GZIPInputStream;


public class MNISTImageDataset extends ImageDataSet {

    public static final Map<String, String> Resources = fillResources();

    private static Map<String, String> fillResources() {
        Map<String, String> resources = new HashMap<>();
        resources.put("train_images", "http://yann.lecun.com/exdb/mnist/train-images-idx3-ubyte.gz");
        resources.put("in_images","http://yann.lecun.com/exdb/mnist/train-labels-idx1-ubyte.gz");
        resources.put("test_images", "http://yann.lecun.com/exdb/mnist/t10k-images-idx3-ubyte.gz");
        resources.put("test_labels","http://yann.lecun.com/exdb/mnist/t10k-labels-idx1-ubyte.gz");
        return Collections.unmodifiableMap(resources);
    }

    private String labelFileName;
    private String imageFileName;

    // the following constants are defined as per the values described at http://yann.lecun.com/exdb/mnist/

    private static final int MAGIC_OFFSET = 0;
    private static final int OFFSET_SIZE = 4; //in bytes

    private static final int LABEL_MAGIC = 2049;
    private static final int IMAGE_MAGIC = 2051;

    private static final int NUMBER_ITEMS_OFFSET = 4;
    private static final int ITEMS_SIZE = 4;

    private static final int NUMBER_OF_ROWS_OFFSET = 8;
    private static final int ROWS_SIZE = 4;
    public static final int ROWS = 28;

    private static final int NUMBER_OF_COLUMNS_OFFSET = 12;
    private static final int COLUMNS_SIZE = 4;
    public static final int COLUMNS = 28;

    private static final int IMAGE_OFFSET = 16;
    private static final int IMAGE_SIZE = ROWS * COLUMNS;


    public MNISTImageDataset() {
        super(28, 28, 1, 10);
    }

    public List<Pair<Integer,float[]>> loadImages() throws IOException {
        return loadImages(imageFileName, labelFileName);
    }

    public List<Pair<Integer,float[]>> loadImages(String... filenames) throws IOException {
        assert (filenames.length % 2 == 0): "expected image and label";

        List<Pair<Integer,float[]>> images = new ArrayList();

        ByteArrayOutputStream labelBuffer = new ByteArrayOutputStream();
        ByteArrayOutputStream imageBuffer = new ByteArrayOutputStream();

        InputStream labelInputStream = new GZIPInputStream(new FileInputStream(filenames[1]));//this.getClass().getResourceAsStream(labelFileName);
        InputStream imageInputStream = new GZIPInputStream(new FileInputStream(filenames[0])); //this.getClass().getResourceAsStream(imageFileName);

        int read;
        byte[] buffer = new byte[16384];

        while((read = labelInputStream.read(buffer, 0, buffer.length)) != -1) {
            labelBuffer.write(buffer, 0, read);
        }

        labelBuffer.flush();

        while((read = imageInputStream.read(buffer, 0, buffer.length)) != -1) {
            imageBuffer.write(buffer, 0, read);
        }

        imageBuffer.flush();

        byte[] labelBytes = labelBuffer.toByteArray();
        byte[] imageBytes = imageBuffer.toByteArray();

        byte[] labelMagic = Arrays.copyOfRange(labelBytes, 0, OFFSET_SIZE);
        byte[] imageMagic = Arrays.copyOfRange(imageBytes, 0, OFFSET_SIZE);

        int magic = ByteBuffer.wrap(labelMagic).getInt();
        if(magic != LABEL_MAGIC)  {
            throw new IOException("Bad magic number in label file got "+magic+"instead of "+LABEL_MAGIC);
        }

        if(ByteBuffer.wrap(imageMagic).getInt() != IMAGE_MAGIC) {
            throw new IOException("Bad magic number in image file!");
        }

        int numberOfLabels = ByteBuffer.wrap(Arrays.copyOfRange(labelBytes, NUMBER_ITEMS_OFFSET, NUMBER_ITEMS_OFFSET + ITEMS_SIZE)).getInt();
        int numberOfImages = ByteBuffer.wrap(Arrays.copyOfRange(imageBytes, NUMBER_ITEMS_OFFSET, NUMBER_ITEMS_OFFSET + ITEMS_SIZE)).getInt();

        if(numberOfImages != numberOfLabels) {
            throw new IOException("The number of labels and images do not match!");
        }

        int numRows = ByteBuffer.wrap(Arrays.copyOfRange(imageBytes, NUMBER_OF_ROWS_OFFSET, NUMBER_OF_ROWS_OFFSET + ROWS_SIZE)).getInt();
        int numCols = ByteBuffer.wrap(Arrays.copyOfRange(imageBytes, NUMBER_OF_COLUMNS_OFFSET, NUMBER_OF_COLUMNS_OFFSET + COLUMNS_SIZE)).getInt();

        if(numRows != ROWS && numCols != COLUMNS) {
            throw new IOException("Bad image. Rows and columns do not equal " + ROWS + "x" + COLUMNS);
        }

        for(int i = 0; i < numberOfLabels; i++) {
            int label = labelBytes[OFFSET_SIZE + ITEMS_SIZE + i];
            byte[] imageData = Arrays.copyOfRange(imageBytes, (i * IMAGE_SIZE) + IMAGE_OFFSET, (i * IMAGE_SIZE) + IMAGE_OFFSET + IMAGE_SIZE);
            float[] imageDataFloat = new float[ROWS * COLUMNS];
            int p = 0;
            for (int j = 0; j < imageData.length; j++) {
                float result = imageData[j] & 0xFF;
                // Convert from [0,255] to [0.0, 1.0]
                result *= 1.0/255.0;
                imageDataFloat[p] = result;
                p++;
            }
            assert p == ROWS * COLUMNS: "Expected: "+ROWS*COLUMNS+" GOT: "+p;
            images.add(new Pair(label, imageDataFloat));
        }

        return images;
    }
}
