package ai.h2o.featurestore.abfs;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.azurebfs.extensions.SASTokenProvider;
import org.apache.hadoop.security.AccessControlException;

/**
 * A class that provides SAS tokens for Feature Store operations on Azure Blob Storage.
 */
public class FeatureStoreSASTokenProvider implements SASTokenProvider {

  private Configuration configuration;

  @Override
  public void initialize(Configuration configuration, String accountName) throws IOException {
    this.configuration = configuration;
  }

  @Override
  public String getSASToken(String account, String fileSystem, String path, String operation)
      throws IOException, AccessControlException {
    String rootToken =
        configuration.get(
                String.format("spark.hadoop.ai.h2o.featurestore.storage.%s.%s.rootSasToken", account, fileSystem));

    String correctedPath = getCorrectedPath(path);
    String token;
    if (correctedPath.isEmpty()) {
      token = rootToken;
    } else {
      Path tokenPath = FileSystems.getDefault().getPath(correctedPath);
      token = findPathToken(account, fileSystem, tokenPath);
    }

    if (token != null && !token.isEmpty()) {
      return token;
    } else {
      return configuration.get("spark.hadoop.ai.h2o.featurestore.storage.staticToken");
    }
  }

  private String getCorrectedPath(String path) {
    String correctedPath = path.replaceAll("/+", "/");
    if (correctedPath.startsWith("/")) {
      return correctedPath.substring(1);
    } else {
      return correctedPath;
    }
  }

  private String findPathToken(String account, String fileSystem, Path path) {
    if (path == null) {
      return configuration.get(
              String.format("spark.hadoop.ai.h2o.featurestore.storage.%s.%s.pathSasToken", account, fileSystem));
    }
    String token =
        configuration.get(
                String.format("spark.hadoop.ai.h2o.featurestore.storage.%s.%s.%s.pathSasToken", account, fileSystem, path));
    if (token != null) {
      return token;
    } else {
      return findPathToken(account, fileSystem, path.getParent());
    }
  }
}
