package ai.h2o.featurestore;

import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.util.Optional;
import org.apache.hadoop.conf.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TokenProvider {

  private final static Logger log = LoggerFactory.getLogger(TokenProvider.class);
  private final Configuration configuration;
  private final String prefix;

  public TokenProvider(Configuration configuration, String prefix) {
    this.configuration = configuration;
    this.prefix = prefix;
  }

  public Token getToken(String path) {
    String rootToken =
        configuration.get(
            String.format("%s.rootSasToken", prefix));

    String correctedPath = getCorrectedPath(path);
    Token token;
    if (correctedPath.isEmpty()) {
      return Token.createToken(rootToken, null);
    } else {
      Path tokenPath = FileSystems.getDefault().getPath(correctedPath);
      token = findPathToken(tokenPath);
    }

    if (token != null) {
      return token;
    } else {
      return Token.createToken(configuration.get("spark.hadoop.ai.h2o.featurestore.storage.staticToken"), null);
    }
  }

  private String getCorrectedPath(String path) {
    String correctedPath = path.replaceAll("/+", "/");
    if (correctedPath.startsWith("/")) {
      return correctedPath.substring(1);
    } else {
      return correctedPath;
    }
  }

  private Token findPathToken(Path path) {
    log.info("Retrieving token for: {}", path);
    if (path == null) {
      String sasToken = configuration.get(prefix + ".pathSasToken");
      String expirationTime = configuration.get(prefix + ".expirationTime");
      return Token.createToken(sasToken, expirationTime);
    }

    return getFilePath(path).or(() -> getDirectoryPath(path)).orElseGet(() -> findPathToken(path.getParent()));
  }

  private Optional<Token> getFilePath(Path path) {
    return getFilePath(path.toString());
  }

  private Optional<Token> getFilePath(String path) {
    String sasToken =
        configuration.get(
            String.format("%s.%s.pathSasToken", prefix, path));
    String expirationTime =
        configuration.get(
            String.format("%s.%s.expirationTime", prefix, path));
    return Optional.ofNullable(Token.createToken(sasToken, expirationTime));
  }

  private Optional<Token> getDirectoryPath(Path path) {
    return getFilePath(path.toString() + "/");
  }

}
