package hex.schemas;

import com.google.gson.Gson;

import java.util.List;
import java.util.Map;
import java.util.Properties;

import hex.Model;
import hex.grid.Grid;
import water.H2O;
import water.Key;
import water.api.API;
import water.api.JobV3;
import water.api.KeyV3;
import water.api.ModelParametersSchema;
import water.api.Schema;
import water.util.IcedHashMap;
import water.util.ReflectionUtils;

/**
 * This is a common grid search schema composed of two parameters: default parameters for a builder
 * and hyper parameters which are given as a mapping from parameter name to list of possible
 * values.
 *
 * @param <G>  a specific implementation type for GridSearch holding results of grid search (model
 *             list)
 * @param <S>  self type
 * @param <MP> actual model parameters type
 * @param <P>  a specific model builder parameters schema, since we cannot derive it from P
 */
public /* FIXME: abstract */ class GridSearchSchema<G extends Grid<MP>,
    S extends GridSearchSchema<G, S, MP, P>,
    MP extends Model.Parameters,
    P extends ModelParametersSchema> extends Schema<G, S> {

  // Hack from ModelBuilderSchema
  public GridSearchSchema() {
    this.parameters = createParametersSchema();
  }

  //
  // Inputs
  //
  @API(help = "Basic model builder parameters.", direction = API.Direction.INPUT)
  public P parameters;

  @API(help = "Grid search parameters.", direction = API.Direction.INOUT)
  public IcedHashMap<String, Object[]> hyper_parameters;

  @API(help = "Destination id for this grid; auto-generated if not specified", required = false, direction = API.Direction.INOUT)
  public KeyV3.GridKeyV3 grid_id;

  //
  // Outputs
  //
  @API(help = "Number of all models generated by grid search.", direction = API.Direction.OUTPUT)
  public int total_models;

  @API(help = "Job Key.", direction = API.Direction.OUTPUT)
  public JobV3 job;

  @Override
  public S fillFromParms(Properties parms) {
    // FIXME: do this in generic way
    if (parms.containsKey("hyper_parameters")) {
      String parameters = parms.getProperty("hyper_parameters");
      hyper_parameters = parseJsonMap(parameters, new IcedHashMap<String, Object[]>());
      parms.remove("hyper_parameters");
    }

    if (parms.containsKey("grid_id")) {
      grid_id = new KeyV3.GridKeyV3(Key.<Grid>make(parms.getProperty("grid_id")));
      parms.remove("grid_id");
    }

    // Do not check validity of
    this.parameters.fillFromParms(parms, false);

    return (S) this;
  }

  @Override
  public S fillFromImpl(G impl) {
    S s = super.fillFromImpl(impl);
    s.parameters = createParametersSchema();
    s.parameters.fillFromImpl((MP) parameters.createImpl());
    return s;
  }

  /**
   * Factory method to create the model-specific parameters schema.
   */
  // FIXME: shared this code with ModelBuilderScheme
  final P createParametersSchema() {

    P impl = null;

    // special case, because GridSearchSchema is the top of the tree and is parameterized differently
    if (GridSearchSchema.class == this.getClass()) {
      return (P) new ModelParametersSchema();
    }

    try {
      Class<? extends ModelParametersSchema>
          parameters_class =
          (Class<? extends ModelParametersSchema>) ReflectionUtils
              .findActualClassParameter(this.getClass(), 3);
      impl = (P) parameters_class.newInstance();
    } catch (Exception e) {
      throw H2O.fail(
          "Caught exception trying to instantiate a builder instance for ModelBuilderSchema: "
          + this + ": " + e, e);
    }
    return impl;
  }

  // this method right now using gson
  // It parses given json, and produces Map<String, Object[]>
  public static <T extends Map<String, Object[]>> T parseJsonMap(String json, T map) {
    Gson gson = new Gson();
    Map<String, Object> m;
    m = gson.fromJson(json, map.getClass());
    for (Map.Entry<String, Object> e : m.entrySet()) {
      if (e.getValue() instanceof List) {
        map.put(e.getKey(), ((List) e.getValue()).toArray());
      } else {
        map.put(e.getKey(), new Object[]{e.getValue()});
      }
    }
    return map;
  }
}
