package water.api;

import water.H2O;
import water.Iced;
import water.IcedWrapper;
import water.Weaver;
import water.exceptions.H2OIllegalArgumentException;
import water.util.Log;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * The metadata info on all the fields in a Schema.  This is used to help Schema be self-documenting,
 * and to generate language bindings for route handlers and entities.
 */
public final class SchemaMetadata extends Iced {

  public int version;
  public String name ;
  public String type;

  public List<FieldMetadata> fields;
  public String markdown;

  // TODO: combine with ModelParameterSchemaV2.
  static public final class FieldMetadata extends Iced {
    /**
     * Field name in the POJO.    Set through reflection.
     */
    String name;

    /**
     * Type for this field.  Set through reflection.
     */
    public String type;

    /**
     * Type for this field is itself a Schema.  Set through reflection.
     */
    public boolean is_schema;

    /**
     * Schema name for this field, if it is_schema.  Set through reflection.
     */
    public String schema_name;

    /**
     * Value for this field.  Set through reflection.
     */
    public Iced value;

    /**
     *  A short help description to appear alongside the field in a UI.  Set from the @API annotation.
     */
    String help;

    /**
     * The label that should be displayed for the field if the name is insufficient.  Set from the @API annotation.
     */
    String label;

    /**
     * Is this field required, or is the default value generally sufficient?  Set from the @API annotation.
     */
    boolean required;

    /**
     * How important is this field?  The web UI uses the level to do a slow reveal of the parameters.  Set from the @API annotation.
     */
    API.Level level;

    /**
     * Is this field an input, output or inout?  Set from the @API annotation.
     */
    API.Direction direction;

    // The following are markers for *input* fields.

    /**
     * For enum-type fields the allowed values are specified using the values annotation.
     * This is used in UIs to tell the user the allowed values, and for validation.
     * Set from the @API annotation.
     */
    String[] values;

    /**
     * Should this field be rendered in the JSON representation?  Set from the @API annotation.
     */
    boolean json;

    /**
     * For Vec-type fields this is the set of Frame-type fields which must contain the named column.
     * For example, for a SupervisedModel the response_column must be in both the training_frame
     * and (if it's set) the validation_frame.
     */
    String[] is_member_of_frames;

    /**
     * For Vec-type fields this is the set of other Vec-type fields which must contain
     * mutually exclusive values.  For example, for a SupervisedModel the response_column
     * must be mutually exclusive with the weights_column.
     */
    String[] is_mutually_exclusive_with;


    public FieldMetadata() { }

    /**
     * @param name field name
     * @param type field type, which can be a primitive type like "string" or "double" or an H2O type like ModelParameters
     * @param value value of the field, represented as a string
     * @param help help text (description) for the field
     * @param label label for the field in the UI, if it should be different from the field name
     * @param required if the field is an INPUT field is it required, or is there a reasonable default value?
     * @param level one of {critical, secondary, expert}: is this a basic field that everyone needs to set, or something only for experts?
     * @param direction one of {INPUT, OUTPUT, INOUT}: is this something the client needs to supply, or something that's only generated by H2O?
     * @param values for enum-type fields this is a list of allowed string values
     * @param json should this field be included in generated JSON?
     */
    public FieldMetadata(String name, String type, boolean is_schema, String schema_name, Iced value, String help, String label, boolean required, API.Level level, API.Direction direction, String[] values, boolean json, String[] is_member_of_frames, String[] is_mutually_exclusive_with) {
      // from the Field, using reflection
      this.name = name;
      this.type = type;
      this.is_schema = is_schema;
      this.schema_name = schema_name;
      this.value = value;

      // from the @API annotation
      this.help = help;
      this.label = label;
      this.required = required;
      this.level = level;
      this.direction = direction;
      this.values = values;
      this.json = json;
      this.is_member_of_frames = is_member_of_frames;
      this.is_mutually_exclusive_with = is_mutually_exclusive_with;
    }

    /**
     * Create a new FieldMetadata object for the given Field of the given Schema.
     * @param schema water.api.Schema object
     * @param f java.lang.reflect.Field for the Schema class
     * @see water.api.SchemaMetadata.FieldMetadata#FieldMetadata(String, String, boolean, String, Iced, String, String, boolean, water.api.API.Level, water.api.API.Direction, String[], boolean, String[], String[])
     */
    public FieldMetadata(Schema schema, Field f) {
      super();
      try {
        this.name = f.getName();
        f.setAccessible(true); // handle private and protected fields
        Object o = f.get(schema);
        this.value = consValue(o);

        boolean is_enum = Enum.class.isAssignableFrom(f.getType());
        this.type = consType(schema, f.getType(), f.getName());
        this.is_schema = (Schema.class.isAssignableFrom(f.getType())) || (f.getType().isArray() && Schema.class.isAssignableFrom(f.getType().getComponentType()));

        // Note, this has to work when the field is null.
        if (this.is_schema) {
          this.schema_name = f.getType().getSimpleName(); // handles arrays as well
        }

        API annotation = f.getAnnotation(API.class);

        if (null != annotation) {
          String l = annotation.label();
          this.help = annotation.help();
          this.label = (null == l || l.isEmpty() ? f.getName() : l);
          this.required = annotation.required();
          this.level = annotation.level();
          this.direction = annotation.direction();
          this.values = annotation.values();
          this.json = annotation.json();
          this.is_member_of_frames = annotation.is_member_of_frames();
          this.is_mutually_exclusive_with = annotation.is_mutually_exclusive_with(); // TODO: need to form the transitive closure

          // If the field is an enum then the values annotation field had better be set. . .
          if (is_enum && (null == this.values || 0 == this.values.length)) {
            throw H2O.fail("Didn't find values annotation for enum field: " + this.name);
          }
        }
      }
      catch (Exception e) {
        throw H2O.fail("Caught exception accessing field: " + f + " for schema object: " + this + ": " + e.toString());
      }
    } // FieldMetadata(Schema, Field)

    /**
     * Factory method to create a new FieldMetadata instance if the Field has an @API annotation.
     * @param schema water.api.Schema object
     * @param f java.lang.reflect.Field for the Schema class
     * @return a new FieldMetadata instance if the Field has an @API annotation, else null
     */
    public static FieldMetadata createIfApiAnnotation(Schema schema, Field f) {
      f.setAccessible(true); // handle private and protected fields

      if (null != f.getAnnotation(API.class))
        return new FieldMetadata(schema, f);

      Log.warn("Skipping field that lacks an annotation: " + schema.toString() + "." + f);
      return null;
    }

    /** For a given Class generate a client-friendly type name (e.g., int[][] or Frame). */
    public static String consType(Schema schema, Class clz, String field_name) {
      boolean is_enum = Enum.class.isAssignableFrom(clz);
      boolean is_array = clz.isArray();

      // built-in Java types:
      if (is_enum)
        return "enum";

      if (String.class.isAssignableFrom(clz))
        return "string"; // lower-case, to be less Java-centric

      if (clz.equals(Boolean.TYPE) || clz.equals(Byte.TYPE) || clz.equals(Short.TYPE) || clz.equals(Integer.TYPE) || clz.equals(Long.TYPE) || clz.equals(Float.TYPE) || clz.equals(Double.TYPE))
        return clz.toString();

      if (is_array)
        return consType(schema, clz.getComponentType(), field_name) + "[]";

      if (Map.class.isAssignableFrom(clz))
        return "Map";

      if (List.class.isAssignableFrom(clz))
        return "List";

      // H2O-specific types:
      // TODO: NOTE, this is a mix of Schema types and Iced types; that's not right. . .
      // Should ONLY have schema types.
      // Also, this mapping could/should be moved to Schema.
      if (water.Key.class.isAssignableFrom(clz)) {
        Log.warn("Raw Key (not KeySchema) in Schema: " + schema.getClass() + " field: " + field_name);
        return "Key";
      }

      if (KeySchema.class.isAssignableFrom(clz)) {
        return "Key<" + KeySchema.getKeyedClassType((Class<? extends KeySchema>)clz) + ">";
      }

      if (Schema.class.isAssignableFrom(clz)) {
        return Schema.getImplClass((Class<Schema>)clz).getSimpleName();  // same as Schema.schema_type
      }

      if (Iced.class.isAssignableFrom(clz)) {
        if (clz == Schema.Meta.class) {
          // Special case where we allow an Iced in a Schema so we don't get infinite meta-regress:
          return "Schema.Meta";
        } else{
          Log.warn("WARNING: found non-Schema Iced field: " + clz.toString() + " in Schema: " + schema.getClass() + " field: " + field_name);
          return clz.getSimpleName();
        }
      }

      String msg = "Don't know how to generate a client-friendly type name for class: " + clz.toString() + " in Schema: " + schema.getClass() + " field: " + field_name;
      Log.warn(msg);
      throw H2O.fail(msg);
    }

    public static Iced consValue(Object o) {
      if (null == o)
        return null;

      Class clz = o.getClass();

      if (water.Iced.class.isAssignableFrom(clz))
        return (Iced)o;

      if (clz.isArray()) {
        return new IcedWrapper(o);
      }

/*
      if (water.Keyed.class.isAssignableFrom(o.getClass())) {
        Keyed k = (Keyed)o;
        return k._key.toString();
      }

      if (! o.getClass().isArray()) {
        if (Schema.class.isAssignableFrom(o.getClass())) {
          return new String(((Schema)o).writeJSON(new AutoBuffer()).buf());
        } else {
          return o.toString();
        }
      }

      StringBuilder sb = new StringBuilder();
      sb.append("[");
      for (int i = 0; i < Array.getLength(o); i++) {
        if (i > 0) sb.append(", ");
        sb.append(consValue(Array.get(o, i)));
      }
      sb.append("]");
      return sb.toString();
      */

      // Primitive type
      if (clz.isPrimitive())
        return new IcedWrapper(o);

      if (o instanceof Number)
        return new IcedWrapper(o);

      if (o instanceof Boolean)
        return new IcedWrapper(o);

      if (o instanceof String)
        return new IcedWrapper(o);

      if (o instanceof Enum)
        return new IcedWrapper(o);


      throw new H2OIllegalArgumentException("o", "consValue", o);
    }

  } // FieldMetadata

  public SchemaMetadata() {
    fields = new ArrayList<>();
  }

  public SchemaMetadata(Schema schema) {
    version = schema.__meta.schema_version;
    name = schema.__meta.schema_name;
    type = schema.__meta.schema_type;

    fields = new ArrayList<>();
    // Fields up to but not including Schema
    for (Field field : Weaver.getWovenFields(schema.getClass())) {
      FieldMetadata fmd = FieldMetadata.createIfApiAnnotation(schema, field);
      if (null != fmd) // skip transient or other non-annotated fields
        fields.add(fmd);  // NOTE: we include non-JSON fields here; remove them later if we don't want them
    }
    this.markdown = schema.markdown(this, null).toString();
  }

  public static SchemaMetadata createSchemaMetadata(String classname) throws IllegalArgumentException {
    try {
      Class<? extends Schema> clz = (Class<? extends Schema>) Class.forName(classname);
      Schema s = clz.newInstance();
      return new SchemaMetadata(s);
    }
    catch (Exception e) {
      String msg = "Caught exception fetching schema: " + classname + ": " + e;
      Log.warn(msg);
      throw new IllegalArgumentException(msg);
    }
  }

}
