/*
 * Decompiled with CFR 0.152.
 */
package water.init;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import water.util.Log;
import water.util.NetworkUtils;
import water.util.OSUtils;

public class HostnameGuesser {
    public static InetAddress findInetAddressForSelf(String ip, String network) throws Error {
        InetAddress local;
        if (ip != null && network != null) {
            throw new HostnameGuessingException("ip and network options must not be used together");
        }
        ArrayList<CIDRBlock> networkList = HostnameGuesser.calcArrayList(network);
        ArrayList<InetAddress> ias = HostnameGuesser.calcPrioritizedInetAddressList();
        if (ip != null) {
            local = HostnameGuesser.getInetAddress(ip, ias);
        } else {
            if (networkList.size() > 0) {
                Log.info("Network list was specified by the user.  Searching for a match...");
                for (InetAddress ia : ias) {
                    Log.info("    Considering " + ia.getHostAddress() + " ...");
                    for (CIDRBlock n : networkList) {
                        if (!n.isInetAddressOnNetwork(ia)) continue;
                        Log.info("    Matched " + ia.getHostAddress());
                        return ia;
                    }
                }
                throw new HostnameGuessingException("No interface matches the network list from the -network option.  Exiting.");
            }
            ArrayList<InetAddress> globalIps = new ArrayList<InetAddress>();
            ArrayList<InetAddress> siteLocalIps = new ArrayList<InetAddress>();
            ArrayList<InetAddress> linkLocalIps = new ArrayList<InetAddress>();
            boolean isIPv6Preferred = NetworkUtils.isIPv6Preferred();
            boolean isIPv4Preferred = NetworkUtils.isIPv4Preferred();
            for (InetAddress ia : ias) {
                if (ia.isLoopbackAddress() || ia.isAnyLocalAddress() || isIPv6Preferred && !isIPv4Preferred && ia instanceof Inet4Address || isIPv4Preferred && ia instanceof Inet6Address) continue;
                if (ia.isSiteLocalAddress()) {
                    siteLocalIps.add(ia);
                }
                if (ia.isLinkLocalAddress()) {
                    linkLocalIps.add(ia);
                }
                globalIps.add(ia);
            }
            local = globalIps.size() == 1 ? (InetAddress)globalIps.get(0) : (siteLocalIps.size() == 1 ? (InetAddress)siteLocalIps.get(0) : (linkLocalIps.size() > 0 ? (InetAddress)linkLocalIps.get(0) : HostnameGuesser.guessInetAddress(siteLocalIps)));
        }
        if (local == null) {
            try {
                Log.warn("Failed to determine IP, falling back to localhost.");
                local = NetworkUtils.isIPv6Preferred() && !NetworkUtils.isIPv4Preferred() ? InetAddress.getByName("::1") : InetAddress.getByName("127.0.0.1");
            }
            catch (UnknownHostException e) {
                Log.throwErr(e);
            }
        }
        return local;
    }

    private static ArrayList<NetworkInterface> calcPrioritizedInterfaceList() {
        ArrayList<NetworkInterface> networkInterfaceList = null;
        try {
            Enumeration<NetworkInterface> nis = NetworkInterface.getNetworkInterfaces();
            ArrayList<NetworkInterface> tmpList = Collections.list(nis);
            Comparator<NetworkInterface> c = new Comparator<NetworkInterface>(){

                @Override
                public int compare(NetworkInterface lhs, NetworkInterface rhs) {
                    if (lhs == null && rhs == null) {
                        return 0;
                    }
                    if (lhs == null) {
                        return 1;
                    }
                    if (rhs == null) {
                        return -1;
                    }
                    if (lhs.getName().equals(rhs.getName())) {
                        return 0;
                    }
                    if (lhs.getName().startsWith("bond") && rhs.getName().startsWith("bond")) {
                        Integer ri;
                        Integer li = lhs.getName().length();
                        if (li.compareTo(ri = Integer.valueOf(rhs.getName().length())) != 0) {
                            return li.compareTo(ri);
                        }
                        return lhs.getName().compareTo(rhs.getName());
                    }
                    if (lhs.getName().startsWith("bond")) {
                        return -1;
                    }
                    if (rhs.getName().startsWith("bond")) {
                        return 1;
                    }
                    return 0;
                }
            };
            Collections.sort(tmpList, c);
            networkInterfaceList = tmpList;
        }
        catch (SocketException e) {
            Log.err(e);
        }
        return networkInterfaceList;
    }

    private static ArrayList<InetAddress> calcPrioritizedInetAddressList() {
        ArrayList<InetAddress> ips = new ArrayList<InetAddress>();
        ArrayList<NetworkInterface> networkInterfaceList = HostnameGuesser.calcPrioritizedInterfaceList();
        boolean isWindows = OSUtils.isWindows();
        boolean isWsl = OSUtils.isWsl();
        int localIpTimeout = NetworkUtils.getLocalIpPingTimeout();
        for (NetworkInterface nIface : networkInterfaceList) {
            Enumeration<InetAddress> ias = nIface.getInetAddresses();
            if (NetworkUtils.isUp(nIface)) {
                while (ias.hasMoreElements()) {
                    InetAddress ia = ias.nextElement();
                    if (isWindows || isWsl || NetworkUtils.isReachable(null, ia, localIpTimeout)) {
                        ips.add(ia);
                        Log.info("Possible IP Address: ", nIface.getName(), " (", nIface.getDisplayName(), "), ", ia.getHostAddress());
                        continue;
                    }
                    Log.info("Network address/interface is not reachable in 150ms: ", ia, "/", nIface);
                }
                continue;
            }
            Log.info("Network interface is down: ", nIface);
        }
        return ips;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     */
    private static InetAddress guessInetAddress(List<InetAddress> ips) {
        InetAddress ip2222222222;
        String m = "Multiple local IPs detected:\n";
        for (InetAddress ip2222222222 : ips) {
            m = m + "  " + ip2222222222;
        }
        m = m + "\nAttempting to determine correct address...\n";
        Socket s = null;
        try {
            s = NetworkUtils.isIPv6Preferred() && !NetworkUtils.isIPv4Preferred() ? new Socket(InetAddress.getByAddress(NetworkUtils.GOOGLE_DNS_IPV6), 53) : new Socket(InetAddress.getByAddress(NetworkUtils.GOOGLE_DNS_IPV4), 53);
            m = m + "Using " + s.getLocalAddress() + "\n";
            ip2222222222 = s.getLocalAddress();
        }
        catch (SocketException se) {
            InetAddress inetAddress = null;
            Log.warn(m);
            if (s != null) {
                try {
                    s.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            return inetAddress;
        }
        catch (Throwable t) {
            Log.err(t);
            InetAddress inetAddress = null;
            {
                catch (Throwable throwable) {
                    Log.warn(m);
                    if (s != null) {
                        try {
                            s.close();
                        }
                        catch (IOException iOException) {
                            // empty catch block
                        }
                    }
                    throw throwable;
                }
            }
            Log.warn(m);
            if (s != null) {
                try {
                    s.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            return inetAddress;
        }
        Log.warn(m);
        if (s != null) {
            try {
                s.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return ip2222222222;
    }

    private static InetAddress getInetAddress(String ip, List<InetAddress> allowedIps) {
        InetAddress addr = null;
        if (ip != null) {
            try {
                addr = InetAddress.getByName(ip);
            }
            catch (UnknownHostException e) {
                throw new HostnameGuessingException(e);
            }
            if (allowedIps != null && !allowedIps.contains(addr)) {
                throw new HostnameGuessingException("IP address not found on this machine");
            }
        }
        return addr;
    }

    static ArrayList<CIDRBlock> calcArrayList(String networkOpt) throws HostnameGuessingException {
        String[] networks;
        ArrayList<CIDRBlock> networkList = new ArrayList<CIDRBlock>();
        if (networkOpt == null) {
            return networkList;
        }
        for (String n : networks = networkOpt.split(",")) {
            CIDRBlock usn = CIDRBlock.parse(n);
            if (usn == null || !usn.valid()) {
                Log.err("Network invalid: " + n);
                throw new HostnameGuessingException("Invalid subnet specification: " + n + " (full '-network' argument: " + networkOpt + ").");
            }
            networkList.add(usn);
        }
        return networkList;
    }

    static class HostnameGuessingException
    extends RuntimeException {
        private HostnameGuessingException(String message) {
            super(message);
        }

        private HostnameGuessingException(Exception e) {
            super(e);
        }
    }

    public static class CIDRBlock {
        private static Pattern NETWORK_IPV4_CIDR_PATTERN = Pattern.compile("(\\d+)\\.(\\d+)\\.(\\d+)\\.(\\d+)/(\\d+)");
        private static Pattern NETWORK_IPV6_CIDR_PATTERN = Pattern.compile("([a-fA-F\\d]+):([a-fA-F\\d]+):([a-fA-F\\d]+):([a-fA-F\\d]+):([a-fA-F\\d]+):([a-fA-F\\d]+):([a-fA-F\\d]+):([a-fA-F\\d]+)/(\\d+)");
        final int[] ip;
        final int bits;

        public static CIDRBlock parse(String cidrBlock) {
            boolean isIPV4 = cidrBlock.contains(".");
            Matcher m = isIPV4 ? NETWORK_IPV4_CIDR_PATTERN.matcher(cidrBlock) : NETWORK_IPV6_CIDR_PATTERN.matcher(cidrBlock);
            boolean b = m.matches();
            if (!b) {
                return null;
            }
            assert (isIPV4 && m.groupCount() == 5 || m.groupCount() == 9);
            int len = isIPV4 ? 4 : 8;
            int[] ipBytes = new int[len];
            for (int i = 0; i < len; ++i) {
                ipBytes[i] = isIPV4 ? Integer.parseInt(m.group(i + 1)) : Integer.parseInt(m.group(i + 1), 16);
            }
            int bits = Integer.parseInt(m.group(len + 1));
            CIDRBlock usn = isIPV4 ? CIDRBlock.createIPv4(ipBytes, bits) : CIDRBlock.createIPv6(ipBytes, bits);
            return usn.valid() ? usn : null;
        }

        public static CIDRBlock createIPv4(int[] ip, int bits) {
            assert (ip.length == 4);
            return new CIDRBlock(ip, bits);
        }

        public static CIDRBlock createIPv6(int[] ip, int bits) {
            assert (ip.length == 8);
            int[] ipLong = new int[16];
            for (int i = 0; i < ip.length; ++i) {
                ipLong[2 * i + 0] = ip[i] >> 8 & 0xFF;
                ipLong[2 * i + 1] = ip[i] & 0xFF;
            }
            return new CIDRBlock(ipLong, bits);
        }

        private CIDRBlock(int[] ip, int bits) {
            assert (ip.length == 4 || ip.length == 16) : "Wrong number of bytes to construct IP: " + ip.length;
            this.ip = ip;
            this.bits = bits;
        }

        private boolean validOctet(int o) {
            return 0 <= o && o <= 255;
        }

        private boolean valid() {
            for (int i = 0; i < this.ip.length; ++i) {
                if (this.validOctet(this.ip[i])) continue;
                return false;
            }
            return 0 <= this.bits && this.bits <= this.ip.length * 8;
        }

        public boolean isInetAddressOnNetwork(InetAddress ia) {
            byte[] ipBytes = ia.getAddress();
            return this.isInetAddressOnNetwork(ipBytes);
        }

        boolean isInetAddressOnNetwork(byte[] ipBytes) {
            int i = 0;
            for (i = 0; i < this.bits / 8; ++i) {
                if ((ipBytes[i] & 0xFF) == this.ip[i]) continue;
                return false;
            }
            int remaining = 0;
            remaining = 8 - this.bits % 8;
            if (remaining < 8) {
                int mask = ~((1 << remaining) - 1) & 0xFF;
                return (ipBytes[i] & 0xFF & mask) == (this.ip[i] & mask);
            }
            return true;
        }
    }
}

