/*
 * Decompiled with CFR 0.152.
 */
package water.persist.security;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.Arrays;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.hadoop.security.token.delegation.AbstractDelegationTokenIdentifier;
import water.H2O;
import water.MRTask;
import water.Paxos;
import water.persist.PersistHdfs;
import water.util.BinaryFileTransfer;
import water.util.FileUtils;

public class HdfsDelegationTokenRefresher
implements Runnable {
    public static final String H2O_AUTH_USER = "h2o.auth.user";
    public static final String H2O_AUTH_PRINCIPAL = "h2o.auth.principal";
    public static final String H2O_AUTH_KEYTAB = "h2o.auth.keytab";
    public static final String H2O_AUTH_TOKEN_REFRESHER_ENABLED = "h2o.auth.tokenRefresher.enabled";
    public static final String H2O_AUTH_TOKEN_REFRESHER_INTERVAL_RATIO = "h2o.auth.tokenRefresher.intervalRatio";
    public static final String H2O_AUTH_TOKEN_REFRESHER_MAX_ATTEMPTS = "h2o.auth.tokenRefresher.maxAttempts";
    public static final String H2O_AUTH_TOKEN_REFRESHER_RETRY_DELAY_SECS = "h2o.auth.tokenRefresher.retryDelaySecs";
    public static final String H2O_AUTH_TOKEN_REFRESHER_FALLBACK_INTERVAL_SECS = "h2o.auth.tokenRefresher.fallbackIntervalSecs";
    public static final String H2O_DYNAMIC_AUTH_S3A_TOKEN_REFRESHER_ENABLED = "h2o.auth.dynamicS3ATokenRefresher.enabled";
    private volatile ScheduledExecutorService _executor;
    private final String _authPrincipal;
    private final String _authKeytabPath;
    private final String _authUser;
    private final double _intervalRatio;
    private final int _maxAttempts;
    private final int _retryDelaySecs;
    private final long _fallbackIntervalSecs;
    private final String _uri;

    public static void setup(Configuration conf, String tmpDir, String uri) throws IOException {
        boolean enabled;
        boolean bl = enabled = conf.getBoolean(H2O_AUTH_TOKEN_REFRESHER_ENABLED, false) || conf.getBoolean(H2O_DYNAMIC_AUTH_S3A_TOKEN_REFRESHER_ENABLED, false);
        if (!enabled) {
            HdfsDelegationTokenRefresher.log("HDFS Token renewal is not enabled in configuration", null);
            return;
        }
        String authUser = conf.get(H2O_AUTH_USER);
        String authPrincipal = conf.get(H2O_AUTH_PRINCIPAL);
        if (authPrincipal == null) {
            HdfsDelegationTokenRefresher.log("Principal not provided, HDFS tokens will not be refreshed by H2O and their lifespan will be limited", null);
            return;
        }
        String authKeytab = conf.get(H2O_AUTH_KEYTAB);
        if (authKeytab == null) {
            HdfsDelegationTokenRefresher.log("Keytab not provided, HDFS tokens will not be refreshed by H2O and their lifespan will be limited", null);
            return;
        }
        String authKeytabPath = HdfsDelegationTokenRefresher.writeKeytabToFile(authKeytab, tmpDir);
        HdfsDelegationTokenRefresher.startRefresher(conf, authPrincipal, authKeytabPath, authUser, uri);
    }

    static void startRefresher(Configuration conf, String authPrincipal, String authKeytabPath, String authUser, String uri) {
        new HdfsDelegationTokenRefresher(conf, authPrincipal, authKeytabPath, authUser, uri).start();
    }

    public static void startRefresher(Configuration conf, String authPrincipal, String authKeytabPath, long renewalIntervalSecs) {
        new HdfsDelegationTokenRefresher(conf, authPrincipal, authKeytabPath, null).start(renewalIntervalSecs);
    }

    private static String writeKeytabToFile(String authKeytab, String tmpDir) throws IOException {
        FileUtils.makeSureDirExists((String)tmpDir);
        File keytabFile = new File(tmpDir, "hdfs_auth_keytab");
        byte[] byteArr = BinaryFileTransfer.convertStringToByteArr((String)authKeytab);
        BinaryFileTransfer.writeBinaryFile((String)keytabFile.getAbsolutePath(), (byte[])byteArr);
        return keytabFile.getAbsolutePath();
    }

    public HdfsDelegationTokenRefresher(Configuration conf, String authPrincipal, String authKeytabPath, String authUser) {
        this(conf, authPrincipal, authKeytabPath, authUser, null);
    }

    public HdfsDelegationTokenRefresher(Configuration conf, String authPrincipal, String authKeytabPath, String authUser, String uri) {
        this._authPrincipal = authPrincipal;
        this._authKeytabPath = authKeytabPath;
        this._authUser = authUser;
        this._intervalRatio = Double.parseDouble(conf.get(H2O_AUTH_TOKEN_REFRESHER_INTERVAL_RATIO, "0.4"));
        this._maxAttempts = conf.getInt(H2O_AUTH_TOKEN_REFRESHER_MAX_ATTEMPTS, 12);
        this._retryDelaySecs = conf.getInt(H2O_AUTH_TOKEN_REFRESHER_RETRY_DELAY_SECS, 10);
        this._fallbackIntervalSecs = conf.getInt(H2O_AUTH_TOKEN_REFRESHER_FALLBACK_INTERVAL_SECS, 43200);
        this._uri = uri;
        this._executor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat(this.getThreadNameFormatForRefresher(conf.getBoolean(H2O_DYNAMIC_AUTH_S3A_TOKEN_REFRESHER_ENABLED, false), uri)).build());
    }

    private String getThreadNameFormatForRefresher(boolean isDynamicS3ATokenRefresherEnabled, String uri) {
        if (isDynamicS3ATokenRefresherEnabled && uri != null) {
            String bucketIdentifier = new Path(uri).toUri().getHost();
            return "s3a-hdfs-token-refresher-" + bucketIdentifier + "-%d";
        }
        return "hdfs-token-refresher-%d";
    }

    void start() {
        long renewalIntervalSecs = this.autodetectRenewalInterval();
        this.start(renewalIntervalSecs);
    }

    void start(long renewalIntervalSecs) {
        if (renewalIntervalSecs <= 0L) {
            throw new IllegalArgumentException("Renewal interval needs to be a positive number, got " + renewalIntervalSecs);
        }
        boolean keepRefreshing = this.doRefresh();
        if (keepRefreshing) {
            assert (this._executor != null) : "Executor is undefined even though we were asked to keep refreshing credentials";
            this._executor.scheduleAtFixedRate(this, renewalIntervalSecs, renewalIntervalSecs, TimeUnit.SECONDS);
        } else {
            HdfsDelegationTokenRefresher.log("Node " + H2O.SELF + " will not be participating in delegation token refresh.", null);
        }
    }

    private long autodetectRenewalInterval() {
        long actualIntervalSecs;
        long intervalSecs = 0L;
        try {
            intervalSecs = this.getTokenRenewalIntervalSecs(this.loginAuthUser());
        }
        catch (IOException | InterruptedException e) {
            HdfsDelegationTokenRefresher.log("Encountered error while trying to determine token renewal interval.", e);
        }
        if (intervalSecs == 0L) {
            actualIntervalSecs = this._fallbackIntervalSecs;
            HdfsDelegationTokenRefresher.log("Token renewal interval was not determined, will use " + this._fallbackIntervalSecs + "s.", null);
        } else {
            actualIntervalSecs = (long)((double)intervalSecs * this._intervalRatio);
            HdfsDelegationTokenRefresher.log("Determined token renewal interval = " + intervalSecs + "s. Using actual interval = " + actualIntervalSecs + "s (ratio=" + this._intervalRatio + ").", null);
        }
        return actualIntervalSecs;
    }

    private static void log(String s, Exception e) {
        System.out.println("HDFS TOKEN REFRESH: " + s);
        if (e != null) {
            e.printStackTrace(System.out);
        }
    }

    @Override
    public void run() {
        boolean keepRefreshing = this.doRefresh();
        if (!keepRefreshing) {
            HdfsDelegationTokenRefresher.log("Cloud is already locked, non-leader node " + H2O.SELF + " will no longer refresh delegation tokens.", null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean doRefresh() {
        if (Paxos._cloudLocked && H2O.CLOUD.leader() != H2O.SELF) {
            if (this._executor != null) {
                ScheduledExecutorService executor;
                HdfsDelegationTokenRefresher hdfsDelegationTokenRefresher = this;
                synchronized (hdfsDelegationTokenRefresher) {
                    executor = this._executor;
                    this._executor = null;
                }
                if (executor != null) {
                    executor.shutdown();
                }
            }
            return false;
        }
        for (int i = 0; i < this._maxAttempts; ++i) {
            try {
                Credentials creds = this.refreshTokens(this.loginAuthUser());
                this.distribute(creds);
                return true;
            }
            catch (IOException | InterruptedException e) {
                HdfsDelegationTokenRefresher.log("Failed to refresh token (attempt " + i + " out of " + this._maxAttempts + "). Will retry in " + this._retryDelaySecs + "s.", e);
                try {
                    Thread.sleep((long)this._retryDelaySecs * 1000L);
                }
                catch (InterruptedException e2) {
                    Thread.currentThread().interrupt();
                }
                continue;
            }
        }
        return true;
    }

    private Credentials refreshTokens(UserGroupInformation tokenUser) throws IOException, InterruptedException {
        return (Credentials)tokenUser.doAs(() -> {
            Credentials creds = new Credentials();
            Object[] tokens = HdfsDelegationTokenRefresher.fetchDelegationTokens(this.getRenewer(), creds, this._uri);
            HdfsDelegationTokenRefresher.log("Fetched delegation tokens: " + Arrays.toString(tokens), null);
            return creds;
        });
    }

    private String getRenewer() {
        return this._authUser != null ? this._authUser : this._authPrincipal;
    }

    private UserGroupInformation loginAuthUser() throws IOException {
        UserGroupInformation realUser;
        HdfsDelegationTokenRefresher.log("Log in from keytab as " + this._authPrincipal, null);
        UserGroupInformation tokenUser = realUser = UserGroupInformation.loginUserFromKeytabAndReturnUGI((String)this._authPrincipal, (String)this._authKeytabPath);
        if (this._authUser != null) {
            HdfsDelegationTokenRefresher.log("Impersonate " + this._authUser, null);
            tokenUser = UserGroupInformation.createProxyUser((String)this._authUser, (UserGroupInformation)tokenUser);
        }
        return tokenUser;
    }

    private long getTokenRenewalIntervalSecs(UserGroupInformation tokenUser) throws IOException, InterruptedException {
        Credentials creds = this.refreshTokens(tokenUser);
        long intervalMillis = (Long)tokenUser.doAs(() -> creds.getAllTokens().stream().map(token -> {
            try {
                long expiresAt = token.renew(PersistHdfs.CONF);
                long issuedAt = 0L;
                TokenIdentifier ident = token.decodeIdentifier();
                if (ident instanceof AbstractDelegationTokenIdentifier) {
                    issuedAt = ((AbstractDelegationTokenIdentifier)ident).getIssueDate();
                }
                return expiresAt - (issuedAt > 0L ? issuedAt : System.currentTimeMillis());
            }
            catch (IOException | InterruptedException e) {
                HdfsDelegationTokenRefresher.log("Failed to determine token expiration for token " + token, e);
                return Long.MAX_VALUE;
            }
        }).min(Long::compareTo).orElse(Long.MAX_VALUE));
        return intervalMillis > 0L && intervalMillis < Long.MAX_VALUE ? intervalMillis / 1000L : 0L;
    }

    private static Token<?>[] fetchDelegationTokens(String renewer, Credentials credentials, String uri) throws IOException {
        if (uri != null) {
            HdfsDelegationTokenRefresher.log("Fetching a delegation token for not-null uri: '" + uri, null);
            return FileSystem.get((URI)URI.create(uri), (Configuration)PersistHdfs.CONF).addDelegationTokens(renewer, credentials);
        }
        return FileSystem.get((Configuration)PersistHdfs.CONF).addDelegationTokens(renewer, credentials);
    }

    private void distribute(Credentials creds) throws IOException {
        DistributeCreds distributeTask = new DistributeCreds(creds);
        if (!Paxos._cloudLocked) {
            distributeTask.setupLocal();
        } else {
            distributeTask.doAllNodes();
        }
    }

    private static class DistributeCreds
    extends MRTask<DistributeCreds> {
        private final byte[] _credsSerialized;

        private DistributeCreds(Credentials creds) throws IOException {
            this._credsSerialized = DistributeCreds.serializeCreds(creds);
        }

        protected void setupLocal() {
            try {
                Credentials creds = this.deserialize();
                HdfsDelegationTokenRefresher.log("Updating credentials", null);
                UserGroupInformation.getCurrentUser().addCredentials(creds);
            }
            catch (IOException e) {
                HdfsDelegationTokenRefresher.log("Failed to update credentials", e);
            }
        }

        private Credentials deserialize() throws IOException {
            ByteArrayInputStream tokensBuf = new ByteArrayInputStream(this._credsSerialized);
            Credentials creds = new Credentials();
            creds.readTokenStorageStream(new DataInputStream(tokensBuf));
            return creds;
        }

        private static byte[] serializeCreds(Credentials creds) throws IOException {
            ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
            DataOutputStream dataStream = new DataOutputStream(byteStream);
            creds.writeTokenStorageToStream(dataStream);
            return byteStream.toByteArray();
        }
    }
}

