package ai.h2o.mojos.runtime.frame;

import java.util.Collection;
import java.util.Map;

/**
 * The builder is used to build rows for a {@link MojoFrameBuilder}. Rows are constructed by adding values in String form to a builder
 * instance. The rows can then be added to a MojoFrameBuilder by calling {@link MojoFrameBuilder#addRow(MojoRowBuilder)}.
 * Any values that are not
 */
public class MojoRowBuilder {
  private final Map<String, Integer> _columnNamesMap;
  private final MojoColumn.Type[] _columnTypes;
  private final Collection<String> _missingValues;

  private Object[] _values;

  MojoRowBuilder(Map<String, Integer> columnNamesMap, MojoColumn.Type[] columnTypes, Collection<String> missingValues) {
    _columnNamesMap = columnNamesMap;
    _columnTypes = columnTypes;
    _missingValues = missingValues;
    _values = new Object[_columnTypes.length];
  }

  /**
   * Set a value to the position associated with column `name` in the row. Nothing happens if `name` is not found in this
   * builder.
   *
   * The `value` is specified as a string and the call will try to convert the value to
   * actual column type.
   *
   *
   * @param name The name of the column to where the value should be set
   * @param value The value to be set
   */
  public void setValue(String name, String value) {
    Integer idx = _columnNamesMap.get(name);
    if (idx != null) {
      setValue(idx, value);
    }
  }

  /**
   * Set a value to an index in the row
   * @param idx The index where the value should be set
   * @param value The value to be set
   */
  public MojoRowBuilder setValue(int idx, String value) {
    if(idx >= _values.length){
      throw new ArrayIndexOutOfBoundsException("Index " + idx + " is out of available scope.");
    }
    _values[idx] = value != null ? (_missingValues.contains(value) ? null : _columnTypes[idx].parse(value)) : null;
    return this;
  }

  public MojoRowBuilder setBool(String name, Boolean value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setBool(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setChar(String name, Character value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setChar(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setByte(String name, Byte value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setByte(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setShort(String name, Short value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setShort(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setInt(String name, Integer value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setInt(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setLong(String name, Long value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setLong(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setFloat(String name, Float value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setFloat(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setDouble(String name, Double value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setDouble(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setString(String name, String value){
    Integer idx = _columnNamesMap.get(name);
    if (idx != null){
      setString(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setDate(String name, java.sql.Date value) {
    Integer idx = _columnNamesMap.get(name);
    if (idx != null) {
      setDate(idx, value);
    }
    return this;
  }

  public MojoRowBuilder setTimestamp(String name, java.sql.Timestamp value) {
    Integer idx = _columnNamesMap.get(name);
    if (idx != null) {
      setTimestamp(idx, value);
    }
    return this;
  }


  private MojoRowBuilder setJavaValue(int idx, Object value){
    if(idx >= _values.length){
      throw new ArrayIndexOutOfBoundsException("Index " + idx + " is out of available scope.");
    }
    _values[idx] = _columnTypes[idx].fromJavaClass(value);
    return this;
  }

  public MojoRowBuilder setBool(int idx, Boolean value) {
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setByte(int idx, Byte value){
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setShort(int idx, Short value){
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setChar(int idx, Character value){
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setInt(int idx, Integer value){
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setLong(int idx, Long value){
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setFloat(int idx, Float value){
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setDouble(int idx, Double value){
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setString(int idx, String value){
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setDate(int idx, java.sql.Date value) {
    return setJavaValue(idx, value);
  }

  public MojoRowBuilder setTimestamp(int idx, java.sql.Timestamp value) {
    return setJavaValue(idx, value);
  }

  /**
   * Set the entire row to `values`.
   *
   * The parameter `values` needs to contain actual object matching types of columns.
   *
   * @param values The array of values to be set into the row.
   */
  public MojoRowBuilder setValues(Object[] values) {
    if (values.length != _values.length)
      throw new IllegalArgumentException("Length of values argument does not match size of MojoRowBuilder");
    System.arraycopy(values, 0, _values, 0, _values.length);
    return this;
  }

  MojoRow toMojoRow() {
    return new MojoRow(_values);
  }

  /**
   * Clear the state of the row builder
   */
  public void clear() {
    _values = new Object[_values.length];
  }

  /**
   * Get the number values a row resulting from this builder would have
   * @return The number of values
   */
  public int size() {
    return _values.length;
  }
}

class MojoRow {

  private Object[] _values;

  MojoRow(Object[] values) {
    _values = values;
  }

  public int size() {
    return _values.length;
  }

  Object[] getValues() {
    return _values;
  }
}
