package ai.h2o.mojos.runtime;

import java.io.IOException;

import ai.h2o.mojos.runtime.lic.LicenseException;
import ai.h2o.mojos.runtime.readers.MojoReaderBackend;
import ai.h2o.mojos.runtime.readers.FolderMojoReaderBackend;

/**
 * Factory class for loading MOJO pipeline objects.
 *
 * "Pipeline" means an entity for generic data transformation. It takes some
 * input data, and produces one or more outputs. The outputs may be arbitrary
 * transforms of the inputs, including new generated features, predictions
 * from some machine learning models, statistical estimates of accuracy, etc.
 *
 * "Mojo" (Model ObJect Optimized) is the H2O's format for saving the models
 * into a file.
 */
public interface MojoPipelineFactory {

  /**
   * A factory method for constructing {@link MojoPipeline} instances from a file.
   *
   * @param file Name of the model's .mojo file.
   *
   * @return a loaded pipeline
   * @throws IOException if <code>file</code> does not exist, or cannot be
   *                     read, or does not represent a valid model.
   */
  Object loadFrom(String file) throws IOException, LicenseException;

  /**
   * Advanced way of constructing instances of {@link MojoPipeline} by supplying a custom
   * mojo reader backend. For example a {@link FolderMojoReaderBackend} can be
   * used to read an unzipped mojo model from a folder. Alternatively, you can
   * implement your own backend, for example to read models stored on network
   * locations, or load them from a database, etc.
   *
   * @param backend a class implementing the {@link MojoReaderBackend}
   *                interface.
   *
   * @return a loaded pipeline
   * @throws IOException if mojo reader fails to read the model, or if the
   *                     model retrieved is not valid.
   */
  Object loadFrom(MojoReaderBackend backend) throws IOException, LicenseException;
}
