package ai.h2o.mojos.runtime.frame;


/**
 * Container for {@link MojoColumn}s as well as some additional meta data. MojoFrames are mostly immutable; it's column/row
 * count, column references, and associated names for said references cannot be modified once created. The data of a
 * individual column can be modified, however, through the `getColumnData` method.
 *
 * MojoFrames can only be constructed through a MojoFrameBuilder (see {@link MojoFrameBuilder} for more details).
 */
public class MojoFrame {

  private final MojoFrameMeta _meta;
  private final MojoColumn[] _columns;
  private final int _nrows;

  
  public void debug() {
    for (int i = 0; i < _columns.length; ++i) {
      System.out.printf("%s\n    %d: %s\n", _meta.getColumnName(i), i, _columns[i].debug());
    }
  }


  MojoFrame(MojoFrameMeta meta, MojoColumn[] columns, int nrows) {
    // Since a frame should be built by a Frame builder we shouldn't need to ensure that nrows
    // matches the length of each column
    _meta = meta;
    _nrows = nrows;
    _columns = columns;
  }


  //----------------------------------------------------------------------------
  // Public API
  //----------------------------------------------------------------------------

  /**
   * Get the number of columns in the frame
   * @return The number of columns in the frame
   */
  public int getNcols() {
    return _columns.length;
  }

  /**
   * Get the number of rows of each column in the MojoFrame
   * @return The number of rows in the frame
   */
  public int getNrows() {
    return _nrows;
  }

  /**
   * Get the names associated with each column in the frame.
   * @return An array containing the given column names for each index
   */
  public String[] getColumnNames() {
    String[] names = new String[_columns.length];
    for (int i = 0; i < names.length; i += 1) {
      names[i] = _meta.getColumnName(i);
    }
    return names;
  }


  /**
   * Get the name of a column at a particular index
   * @param index
   * @return The name of the column at the given index
   */
  public String getColumnName(int index) {
    return _meta.getColumnName(index);
  }

  /**
   * Get the type of a column at a particular index
   * @param index
   * @return The type of the column at the given index
   */
  public MojoColumn.Type getColumnType(int index) {
    return _meta.getColumnType(index);
  }

  /**
   * Get the types of each column in the frame
   * @return An array containing the column types for each index
   */
  public MojoColumn.Type[] getColumnTypes() {
    MojoColumn.Type[] types = new MojoColumn.Type[_columns.length];
    for (int i = 0; i < _columns.length; i++) {
      types[i] = _columns[i].getType();
    }
    return types;
  }
  
  /**
   * Get the column instance at a particular index
   * @param index
   * @return The column instance at the given index
   */
  public MojoColumn getColumn(int index) {
    return _columns[index];
  }

  /**
   * Get the data stored in the column at a particular index
   * @param index
   * @return The data of a column at the given index. This will be an array of whatever java type the column's Type
   * is intended to represent (see {@link MojoColumn.Type})
   */

  public Object getColumnData(int index) { return _columns[index].getData(); }

  MojoColumn[] getColumns() { return _columns; }
}
