package ai.h2o.mojos.runtime.readers;

import java.io.*;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import static ai.h2o.mojos.runtime.readers.MojoReaderBackend.*;

/**
 * The backend serves MOJO content from in memory file system.
 */
public class InMemoryMojoReaderBackend extends MojoReaderBackend {

  private static final Map<String, byte[]> CLOSED = Collections.unmodifiableMap(new HashMap<String, byte[]>());

  private Map<String, byte[]> _mojoContent;

  public InMemoryMojoReaderBackend(Map<String, byte[]> mojoContent) {
    this(mojoContent, null);
  }
  
  public InMemoryMojoReaderBackend(Map<String, byte[]> mojoContent, String baseDir) {
    this(mojoContent, baseDir, null);
  }

  public InMemoryMojoReaderBackend(Map<String, byte[]> mojoContent, String baseDir, String separator) {
    this(mojoContent, baseDir, separator, null);
  }

  public InMemoryMojoReaderBackend(Map<String, byte[]> mojoContent, String baseDir,
                                   String separator, String pipelineFileName) {
    super(baseDir, separator, pipelineFileName);
    _mojoContent = mojoContent;
  }

  @Override
  public InputStream getFile(String filename) throws IOException {
    assertOpen();
    byte[] data = getBinaryFile(filename);
    if (data == null)
      throw new IOException("MOJO doesn't contain resource " + filename);
    return new ByteArrayInputStream(data);
  }

  @Override
  public BufferedReader getTextFile(String filename) throws IOException {
    assertOpen();
    return new BufferedReader(new InputStreamReader(getFile(filename)));
  }

  @Override
  public byte[] getBinaryFile(String filename) {
    assertOpen();
    return _mojoContent.get(filenameToPath(filename));
  }

  @Override
  public boolean exists(String filename) {
    return _mojoContent.containsKey(filenameToPath(filename));
  }


  @Override
  public void close() {
    _mojoContent = CLOSED;
  }

  private void assertOpen() {
    if (_mojoContent == CLOSED)
      throw new IllegalStateException("ReaderBackend was already closed");
  }

  private enum PipelineType {
    NONE,
    TOML,
    PROTO,
  }

  static InMemoryMojoReaderBackend createFrom(InputStream inputStream) throws IOException {
    HashMap<String, byte[]> content = new HashMap<>();
    ZipInputStream zis = new ZipInputStream(inputStream);
    PipelineType pipelineType = PipelineType.NONE;
    try {
      ZipEntry entry;
      while ((entry = zis.getNextEntry()) != null) {
        if (entry.getSize() > Integer.MAX_VALUE)
          throw new IOException("File too large: " + entry.getName());
        if (!entry.isDirectory()) {
          String entryFilename = entry.getName();
          if (entryFilename.equals(DEFAULT_PROTO_PIPELINE_FILE_PATH)) {
            pipelineType = PipelineType.PROTO;
          } else if (entryFilename.equals(DEFAULT_TOML_PIPELINE_FILE_PATH)) {
            pipelineType = PipelineType.TOML;
          }
        }
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        copyStream(zis, os);
        content.put(entry.getName(), os.toByteArray());
      }
    } finally {
      closeQuietly(zis);
    }
    // Now create the right backend
    switch (pipelineType) {
      case TOML:
        return new InMemoryMojoReaderBackend(content, DEFAULT_BASE_DIR, "/", DEFAULT_TOML_PIPELINE_FILENAME);
      case PROTO:
        return new InMemoryMojoReaderBackend(content);
      default:
        throw new IOException("Cannot find any pipeline file!");
    }
  }

  private static void closeQuietly(Closeable c) {
    if (c != null)
      try {
        c.close();
      } catch (IOException e) {
        // intentionally ignore exception
      }
  }
  private static void copyStream(InputStream source, OutputStream target) throws IOException {
    byte[] buffer = new byte[8 * 1024];
    while (true) {
      int len = source.read(buffer);
      if (len == -1)
        break;
      target.write(buffer, 0, len);
    }
  }

  private String filenameToPath(String filename) {
    return getBaseDir() + filename;
  }

}
