package ai.h2o.mojos.runtime.frame;

import java.util.ArrayList;
import java.util.List;
import java.util.ServiceLoader;

/**
 * Service to obtain instance of {@link MojoColumnFactory}.
 *
 * The service is using Java SPI to obtain all classes
 * implementing {@link MojoColumnFactory} interface.
 *
 * https://en.wikipedia.org/wiki/Initialization-on-demand_holder_idiom#Example_Java_Implementation
 */
final class MojoColumnFactoryService {

  private static class InstanceHolder {
    private static MojoColumnFactoryService INSTANCE = new MojoColumnFactoryService();
  }

  public static MojoColumnFactoryService getInstance() {
    return InstanceHolder.INSTANCE;
  }

  private final MojoColumnFactory mojoColumnFactory;
  private final RuntimeException error;

  private MojoColumnFactoryService() {
    ServiceLoader<MojoColumnFactory> loader = ServiceLoader.load(MojoColumnFactory.class);
    MojoColumnFactory[] factories = getAll(loader);
    if (factories.length == 0) {
      mojoColumnFactory = null;
      error = new RuntimeException("Cannot find MOJO column factory implementation! Check the classpath if it contains mojo2-runtime-impl!");
    } else if (factories.length > 1) {
      mojoColumnFactory = null;
      error = new RuntimeException("Found multiple MOJO column factories implementation backends, but expected only one! Check the classpath if it contains mojo2-runtime-impl!");
    } else {
      error = null;
      mojoColumnFactory = factories[0];
    }
  }

  public MojoColumnFactory getMojoColumnFactory() {
    if (mojoColumnFactory != null) {
      return mojoColumnFactory;
    } else {
      throw error;
    }
  }

  /**
   * Get all implementors of {@link MojoColumnFactory} interface.
   *
   * @return array of objects implementing {@link MojoColumnFactory} interface. Empty array
   * if no implementor is found.
   */
  private static MojoColumnFactory[] getAll(ServiceLoader<MojoColumnFactory> loader) {
    List<MojoColumnFactory> l = new ArrayList<>();
    for (MojoColumnFactory mcf : loader) {
      l.add(mcf);
    }
    return l.toArray(new MojoColumnFactory[0]);
  }
}
