package ai.h2o.mojos.runtime.frame;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;


/**
 * MojoFrame meta data
 * ===========================
 *
 * Container for meta data of a {@link MojoFrame}. This immutable class is used as a template for constructing
 * MojoFrames in addition to providing names to the columns of an existing MojoFrame.
 */
public class MojoFrameMeta implements Serializable {
  private String[] _columnNames;
  private MojoColumn.Type[] _columnTypes;
  private Map<String, Integer> _columnNameToIndex;

  /**
   * Construct a MojoFrameMeta instance with the given column names and types. The length of the two array arguments
   * must be equal.
   * @param columnNames The names of the columns
   * @param columnTypes The types of the columns
   */
  public MojoFrameMeta(String[] columnNames, MojoColumn.Type[] columnTypes) {
    if (columnNames.length != columnTypes.length)
      throw new IllegalArgumentException("columnNames and columnTypes arguments must have the same length");
    _columnNameToIndex = new HashMap<>();
    for (int i = 0; i < columnNames.length; i += 1) {
      if (_columnNameToIndex.containsKey(columnNames[i]))
        throw new IllegalArgumentException("Columns may not have duplicate names: \"" +
            columnNames[i] + "\"");
      _columnNameToIndex.put(columnNames[i], i);
    }
    _columnNames = new String[columnNames.length];
    _columnTypes = new MojoColumn.Type[columnTypes.length];
    System.arraycopy(columnNames, 0, _columnNames, 0, _columnNames.length);
    System.arraycopy(columnTypes, 0, _columnTypes, 0, _columnTypes.length);
  }


  /**
   * Get the index of a column with the name `columnName`
   * @param columnName The name of the column
   * @return The index of the column
   */
  public int getColumnIndex(String columnName) {
    return _columnNameToIndex.get(columnName);
  }

  /**
   * Get the name of a column at a particular index
   * @param index The index of the column
   * @return The name of the column
   */
  public String getColumnName(int index) {
    return _columnNames[index];
  }

  /**
   * Get the type of a column at a particular index
   * @param index The index of a column
   * @return The type of the column
   */
  public MojoColumn.Type getColumnType(int index) {
    return _columnTypes[index];
  }


  /**
   * Get the type of a column with the name `columnName`
   * @param columnName The name of the column
   * @return The type of the column
   */
  public MojoColumn.Type getColumnType(String columnName) {
    return _columnTypes[getColumnIndex(columnName)];
  }

  /**
   * Checks if there exists meta data for a column with a particular name
   * @param columnName The name of the column
   * @return true if the name exists in this instance, false otherwise
   */
  public boolean contains(String columnName) {
    return _columnNameToIndex.containsKey(columnName);
  }

  /**
   * Get the number of columns in this instance
   * @return The number of columns in this instance
   */
  public int size() { return _columnNames.length; }

  protected Map<String, Integer> getColumnNamesMap() {
    return _columnNameToIndex;
  }

  public String[] getColumnNames() {
    return _columnNames;
  }

  public MojoColumn.Type[] getColumnTypes() {
    return _columnTypes;
  }



  /**
   * Make a MojoFrameMeta instance with no columns
   * @return A MojoFrameMeta instance with no columns
   */
  public static MojoFrameMeta getEmpty() {
    return new MojoFrameMeta(new String[0], new MojoColumn.Type[0]);
  }

}
