package ai.h2o.mojos.runtime.readers;

import java.io.*;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * The backend serves MOJO content from a ZIP file.
 */
public class MojofileMojoReaderBackend extends MojoReaderBackend {
  private final ZipFile _zf;


  public MojofileMojoReaderBackend(String archivename) throws IOException {
    this(archivename,  null);
  }

  public MojofileMojoReaderBackend(String archivename, String baseDir) throws IOException {
    this(archivename, baseDir, null);
  }

  public MojofileMojoReaderBackend(String archivename, String baseDir, String pipelineFileName) throws IOException {
    super(baseDir, "/", pipelineFileName);
    _zf = new ZipFile(archivename);
  }

  @Override
  public InputStream getFile(String filename) throws IOException {
    return _zf.getInputStream(getEntry(filename));
  }

  @Override
  public BufferedReader getTextFile(String filename) throws IOException {
    InputStream input = getFile(filename);
    return new BufferedReader(new InputStreamReader(input));
  }

  @Override
  public byte[] getBinaryFile(String filename) throws IOException {
    ZipEntry entry = getEntry(filename);
    byte[] out = new byte[(int) entry.getSize()];
    DataInputStream dis = new DataInputStream(_zf.getInputStream(entry));
    dis.readFully(out);
    return out;
  }

  @Override
  public String[] getFileNames(String path) {
    path = filenameToPath(path);
    if (!(path.isEmpty() || path.endsWith(getSeparator()))) {
      path += getSeparator();
    }
    Set<String> fileNames = new HashSet<>();
    Enumeration<? extends ZipEntry> entries = _zf.entries();
    while (entries.hasMoreElements()) {
      ZipEntry entry = entries.nextElement();
      String entryName = entry.getName();
      if (entryName.startsWith(path)) {
        String f = entryName.substring(path.length());
        if (!f.contains(getSeparator())) {
          fileNames.add(f);
        }
      }
    }
    String[] out = new String[fileNames.size()];
    fileNames.toArray(out);
    return out;
  }

  @Override
  public String[] getDirectoryNames(String path) {
    path = filenameToPath(path);
    if (!(path.isEmpty() || path.endsWith(getSeparator()))) {
      path += getSeparator();
    }
    Set<String> dirNames = new HashSet<>();
    Enumeration<? extends ZipEntry> entries = _zf.entries();
    while (entries.hasMoreElements()) {
      ZipEntry entry = entries.nextElement();
      String entryName = entry.getName();
      if (entryName.startsWith(path)) {
        int dirEnd = entryName.indexOf(getSeparator(), path.length());
        if (dirEnd != -1) {
          dirNames.add(entryName.substring(path.length(), dirEnd + getSeparator().length()));
        }
      }
    }
    String[] out = new String[dirNames.size()];
    dirNames.toArray(out);
    return out;
  }

  @Override
  public void close() throws IOException {
    _zf.close();
  }

  @Override
  public boolean exists(String filename) {
    return _zf.getEntry(filenameToPath(filename)) != null;
  }

  private ZipEntry getEntry(String filename) throws IOException {
    ZipEntry ze = _zf.getEntry(filenameToPath(filename));
    if (ze == null)
      throw new IOException("Tree file '" + filenameToPath(filename) + "' not found");
    return ze;
  }

  private String filenameToPath(String filename) {
    return getBaseDir() + filename;
  }

}
