package ai.h2o.mojos.runtime.frame;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * The builder is used for constructing a {@link MojoFrame}. A frame is constructed by the following procedure:
 *
 *   1. Get a MojoRowBuilder instance from the frame builder
 *   2. Construct a row from the MojoRowBuilder
 *   3. Append the resulting row to the frame builder
 *   4. Repeat steps 1-3 until all rows are constructed
 *   5. Construct the MojoFrame
 *
 * See {@link MojoRowBuilder}
 */
public class MojoFrameBuilder {
  public static final StringConverter DEFAULT_CONVERTER = new StringConverter() {
    @Override
    public Object convert(String s, MojoColumn.Type outputType) {
      return outputType.parse(s);
    }
  };
  private final MojoFrameMeta _meta;
  private final Set<String> _missingValues;
  private final MojoColumnBuilder[] _columnBuilders;
  private final StringConverter[] _stringConverters;

  /**
   * Constructor for a MojoFrameBuilder.
   *
   * @param frameMeta The meta data for the resulting frame (see {@link MojoFrameMeta})
   * @param missingValues  List of string values which are interpreted as missing value.
   */
  public MojoFrameBuilder(MojoFrameMeta frameMeta, Collection<String> missingValues) {
    this(frameMeta, missingValues, new HashMap<String, StringConverter>(0));
  }

  /**
   * Constructor for a MojoFrameBuilder.
   *
   * @param frameMeta The meta data for the resulting frame (see {@link MojoFrameMeta})
   * @param missingValues  List of string values which are interpreted as missing value.
   * @param stringConverters A `Map` that associates column names to their respective {@link StringConverter}.
   *                         `DEFAULT_CONVERTER` is used if a column's name is not found in the `Map`
   */
  public MojoFrameBuilder(MojoFrameMeta frameMeta, Collection<String> missingValues,
                          Map<String, StringConverter> stringConverters) {
    _meta = frameMeta;
    MojoColumn.Type[] columnTypes = frameMeta.getColumnTypes();
    String[] columnNames = frameMeta.getColumnNames();
    if (missingValues != null) {
      _missingValues = new HashSet<>(missingValues);
    } else {
      _missingValues = new HashSet<>(0);
    }
    _columnBuilders = new MojoColumnBuilder[frameMeta.size()];
    _stringConverters = new StringConverter[frameMeta.size()];
    for (int i = 0; i < _columnBuilders.length; i += 1) {
      _columnBuilders[i] = new MojoColumnBuilder(columnTypes[i]);
      String name = columnNames[i];
      _stringConverters[i] = stringConverters.containsKey(name) ? stringConverters.get(name) : DEFAULT_CONVERTER;
    }
  }

  /**
   * Constructor for a MojoFrameBuilder.
   *
   * @param frameMeta The meta data for the resulting frame (see {@link MojoFrameMeta})
   */
  public MojoFrameBuilder(MojoFrameMeta frameMeta) {
    this(frameMeta, Collections.<String>emptyList());
  }

  /**
   * Append a row from the current state of a MojoRowBuilder. The MojoRowBuilder will subsequently be reset.
   * @param rowBuilder The MojoRowBuilder containing the row to be constructed and appended
   * @return The given MojoRowBuilder instance with its state reset
   */
  public MojoRowBuilder addRow(MojoRowBuilder rowBuilder) {
    addRow(rowBuilder.toMojoRow());
    rowBuilder.clear();
    return rowBuilder;
  }

  void addRow(MojoRow row) {
    Object[] values = row.getValues();
    if (values.length != _columnBuilders.length)
      throw new IllegalArgumentException("Row argument does not have the same column count as frame");
    for (int i = 0; i < _columnBuilders.length; i += 1) {
      _columnBuilders[i].pushValue(values[i]);
    }
  }

  /**
   * Get an instance of a MojoRowBuilder that can be used to construct a row for this builder. Each call to this method
   * creates a new MojoRowBuilder instance
   * @return A MojoRowBuilder for constructing rows for this frame builder
   */
  public MojoRowBuilder getMojoRowBuilder() {
    return getMojoRowBuilder(false);
  }


  /**
   * Get an instance of a MojoRowBuilder that can be used to construct a row for this builder. Each call to this method
   * creates a new MojoRowBuilder instance
   * @param strictMode flag to determine if the created MojoRowBuilder should be in "strict" mode (see {@link MojoRowBuilder}).
   * @return A MojoRowBuilder for constructing rows for this frame builder
   */
  public MojoRowBuilder getMojoRowBuilder(boolean strictMode) {
    return new MojoRowBuilder(_meta.getColumnNamesMap(), _meta.getColumnTypes(), _missingValues, _stringConverters, strictMode);
  }


  /**
   * Create a MojoFrame from the current state of this builder
   * @return The constructed MojoFrame
   */
  public MojoFrame toMojoFrame() {
    MojoColumn[] columns = new MojoColumn[_columnBuilders.length];
    int nrows = _columnBuilders.length == 0 ? 0 : _columnBuilders[0].size();
    for (int i = 0; i < columns.length; i += 1) {
      columns[i] = _columnBuilders[i].toMojoColumn();
    }
    return new MojoFrame(_meta, columns, nrows);
  }


  /**
   * Create a MojoFrame with `nrows` rows based on the meta data provided. The values in this frame will all be NA.
   * @param meta The meta data of the frame to be constructed
   * @param nrows The number of rows
   * @return A new MojoFrame filled with NA values
   */
  public static MojoFrame getEmpty(MojoFrameMeta meta, int nrows) {
    MojoColumn.Type[] types = meta.getColumnTypes();
    MojoColumn[] columns = new MojoColumn[meta.size()];
    for (int i = 0; i < meta.size(); i += 1) {
      columns[i] = MojoColumnFactoryService.getInstance().getMojoColumnFactory().create(types[i], nrows);
    }
    return new MojoFrame(meta, columns, nrows);
  }

  /**
   * Create a MojoFrame in accordance to a {@link MojoFrameMeta} by concatenating a series of existing MojoFrames. This
   * means that columns are added to the return frame based on position and not by name. The total number of columns
   * across all of the argument frames must match the size of the meta data, and each argument frame must have the same
   * number of rows.
   * The resulting frame contains references to the same column instances
   * @param meta The meta data to use as a template for the return frame
   * @param frames The frame(s) to be used to construct the return frame
   * @return A MojoFrame containg references to the columns in the argument frames
   */
  public static MojoFrame fromFrames(MojoFrameMeta meta, MojoFrame... frames) {
    int ncols = 0;
    int nrows = frames.length == 0 ? 0 : frames[0].getNrows();
    for (MojoFrame f : frames) {
      if (f.getNrows() != nrows && f.getNcols() != 0)
        throw new IllegalArgumentException("Number of rows in frames do not match");
      ncols += f.getNcols();
    }
    if (ncols != meta.size())
      throw new IllegalArgumentException("Total number of columns in frames does not equal size of frame meta");
    MojoColumn[] columns = new MojoColumn[ncols];
    MojoColumn.Type[] types = meta.getColumnTypes();
    int curr = 0;
    for (MojoFrame f : frames) {
      for (int i = 0; i < f.getNcols(); i += 1) {
        MojoColumn col = f.getColumn(i);
        if (col.getType() != types[curr])
          throw new IllegalArgumentException("Type of column " + curr + " does not match frame meta");
        columns[curr] = f.getColumn(i);
        curr += 1;
      }
    }
    return new MojoFrame(meta, columns, nrows);
  }

  /**
   * Create a MojoFrame from an array of MojoColumns as specified by the provided meta data.
   * @param meta The meta data to be used as a template
   * @param columns The columns to be used in the resulting frame
   * @return A new MojoFrame
   */
  public static MojoFrame fromColumns(MojoFrameMeta meta, MojoColumn[] columns) {
    if (columns.length != meta.size())
      throw new IllegalArgumentException("Number of columns does not match size of frame meta");
    int nrows = columns.length == 0 ? 0 : columns[0].size();
    MojoColumn.Type[] types = meta.getColumnTypes();
    for (int i = 0; i < columns.length; i += 1) {
      MojoColumn c = columns[i];
      if (c.size() != nrows)
        throw new IllegalArgumentException("Number of rows in columns do not match");
      if (c.getType() != types[i])
        throw new IllegalArgumentException("Type of column " + i + " does not match frame meta");
    }
    return new MojoFrame(meta, columns, nrows);
  }

  private static class MojoColumnBuilder {
    private final MojoColumn.Type colType;
    private final List<Object> values = new ArrayList<>();

    MojoColumnBuilder(MojoColumn.Type type) {
      colType = type;
    }

    void pushValue(Object value) {
      values.add(value == null ? colType.NULL : value);
    }

    MojoColumn toMojoColumn() {
      final MojoColumn col = MojoColumnFactoryService.getInstance().getMojoColumnFactory().create(colType, values.size());
      col.fillFromParsedListData(values);
      return col;
    }

    int size() {
      return values.size();
    }
  }
}


