package ai.h2o.mojos.runtime.api.backend;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/** Access to in-memory fileset; primarily for testing */
public class MemoryReaderBackend implements ReaderBackend {
    private final Map<String, byte[]> files;

    private MemoryReaderBackend(Map<String, byte[]> files) {
        this.files = files;
    }

    public static ReaderBackend open(Map<String, byte[]> files) {
        return new MemoryReaderBackend(files);
    }

    /**
     * Read zipped files from inputstream. Useful when the container is nested in a java resource.
     */
    public static ReaderBackend fromZipStream(final InputStream is) throws IOException {
        final HashMap<String, byte[]> files = new HashMap<>();
        try (final ZipInputStream zis = new ZipInputStream(is)) {
            while (true) {
                final ZipEntry entry = zis.getNextEntry();
                if (entry == null) {
                    break;
                }
                if (entry.getSize() > Integer.MAX_VALUE) {
                    throw new IOException("File is too large: " + entry.getName());
                }
                if (!entry.isDirectory()) {
                    final ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    ReaderBackendUtils.copy(zis, baos, 8 * 1024);
                    files.put(entry.getName(), baos.toByteArray());
                }
                zis.closeEntry();
            }
        }
        return MemoryReaderBackend.open(files);
    }

    public void put(final String resourceName, final byte[] bytes) {
        files.put(resourceName, bytes);
    }

    @Override
    public InputStream getInputStream(String resourceName) throws FileNotFoundException {
        final byte[] bytes = files.get(resourceName);
        if (bytes == null) {
            throw new FileNotFoundException(resourceName);
        }
        return new ByteArrayInputStream(bytes);
    }

    @Override
    public boolean exists(String resourceName) {
        return files.containsKey(resourceName);
    }

    @Override
    public Collection<String> list() {
        return files.keySet();
    }

    @Override
    public void close() {}

    @Override
    public String toString() {
        return String.format("%s[%s]", getClass().getSimpleName(), files.size());
    }
}
