package ai.h2o.mojos.runtime.api;

import ai.h2o.mojos.runtime.frame.MojoColumn;
import java.util.ArrayList;
import java.util.List;

/**
 * Represents all metadata about a column.
 * It can be input feature, intermediate result, or output column.
 *
 * Does NOT contain values - for that, see {@link ai.h2o.mojos.runtime.frame.MojoColumn} and its descendants.
 */
public class MojoColumnMeta {
    private final String columnName;
    private final MojoColumn.Type columnType;
    private final MojoColumn.Kind kind;

    public /*todo: private*/ MojoColumnMeta(String columnName, MojoColumn.Type columnType, MojoColumn.Kind kind) {
        this.columnName = columnName;
        this.columnType = columnType;
        if (kind == MojoColumn.Kind.Response) {
            throw new IllegalArgumentException(String.format("Column '%s':%s cannot be created, invalid kind used: '%s'", columnName, columnType, kind));
        }
        this.kind = kind;
    }

    public static MojoColumnMeta newInput(String columnName, MojoColumn.Type columnType) {
        return new MojoColumnMeta(columnName, columnType, MojoColumn.Kind.Feature);
    }

    public static MojoColumnMeta newOutput(String name, MojoColumn.Type columnType) {
        return new MojoColumnMeta(name, columnType, MojoColumn.Kind.Output);
    }

    public static MojoColumnMeta newInterim(String name, MojoColumn.Type colType) {
        return new MojoColumnMeta(name, colType, MojoColumn.Kind.Interim);
    }

    /**
     * Compatibility helper. You should preferably construct {@link List} of columns as standard java collection, and then pass it to {@link ai.h2o.mojos.runtime.frame.MojoFrameMeta}.
     */
    public static List<MojoColumnMeta> toColumns(final String[] columnNames, final MojoColumn.Type[] columnTypes, final MojoColumn.Kind kind) {
        if (columnNames.length != columnTypes.length) {
            throw new IllegalArgumentException("columnNames and columnTypes arguments must have the same length");
        }
        final List<MojoColumnMeta> cols = new ArrayList<>();
        for (int i = 0; i < columnNames.length; i++) {
            final MojoColumnMeta col = new MojoColumnMeta(columnNames[i], columnTypes[i], kind);
            cols.add(col);
        }
        return cols;
    }

    public String getColumnName() {
        return columnName;
    }

    public MojoColumn.Type getColumnType() {
        return columnType;
    }

    /**
     * @return true if this column represents an input of the entire pipeline.
     */
    public boolean isInputFeature() {
        return kind == MojoColumn.Kind.Feature;
    }

    /**
     * @return true if this column represents final result of the entire pipeline.
     */
    public boolean isOutputColumn() {
        return kind == MojoColumn.Kind.Output;
    }

    @Override
    public String toString() {
        return String.format("%s{name:'%s',type=%s}", kind.toString(), columnName, columnType);
    }
}
