package ai.h2o.mojos.runtime.transforms;

import ai.h2o.mojos.runtime.api.MojoTransformationGroup;
import ai.h2o.mojos.runtime.frame.MojoColumn;
import ai.h2o.mojos.runtime.frame.MojoFrame;
import ai.h2o.mojos.runtime.frame.MojoFrameMeta;
import java.util.Arrays;

/**
 * Representation of one transformation operation in the pipeline.
 */
public abstract class MojoTransform {

    public final int[] iindices;
    public final int[] oindices;
    public final MojoFrameMeta meta;
    private String id;
    private String name;
    private MojoTransformationGroup transformationGroup;

    public MojoTransform(MojoFrameMeta meta, int[] iindices, int[] oindices) {
        assert iindices != null;
        assert oindices != null;
        this.meta = meta;
        this.iindices = iindices;
        this.oindices = oindices;
    }

    public MojoColumn.Type[] getInputTypes() {
        MojoColumn.Type[] types = new MojoColumn.Type[iindices.length];
        for (int i = 0; i < types.length; i += 1) {
            types[i] = meta.getColumnType(iindices[i]);
        }
        return types;
    }

    public MojoColumn.Type getInputType(int idx) {
        return meta.getColumnType(iindices[idx]);
    }

    public MojoColumn.Type[] getOutputTypes() {
        MojoColumn.Type[] types = new MojoColumn.Type[oindices.length];
        for (int i = 0; i < types.length; i += 1) {
            types[i] = meta.getColumnType(oindices[i]);
        }
        return types;
    }

    public MojoColumn.Type getOutputType(int idx) {
        return meta.getColumnType(oindices[idx]);
    }

    /**
     * <p>Hierarchical transformer index identification.</p>
     *
     * To compute the identification, each transformer in the pipeline receives an index based on its physical order, starting from 0.<br/>
     * This index is formatted with zero-padded 3 digit format, and this forms one component in the hierarchical index.<br/>
     * Top-level transforms are prefixed with letter <code>T</code>, indicating the root pipeline as its container.<br/>
     * Nested transforms (those inside <code>EXEC_OP</code> contained) use the container's ID as its prefix, followed by underscore, followed by own index.<br/>
     *
     * Examples:
     * <ul>
     *     <li><code>T000</code> = first transform in the root pipeline</li>
     *     <li><code>T005_032</code> = in the root pipeline's EXEC_OP at position 005, the nested pipeline has this transform at position 032</li>
     * </ul>
     */
    public String getId() {
        return id;
    }

    public void setId(final String id) {
        this.id = id;
    }

    public MojoTransformationGroup getTransformationGroup() {
        return transformationGroup;
    }

    public void setTransformationGroup(final MojoTransformationGroup transformationGroup) {
        this.transformationGroup = transformationGroup;
    }

    public String getName() {
        return name;
    }

    public void setName(final String name) {
        this.name = name;
    }

    public abstract void transform(MojoFrame frame);

    @Override
    public String toString() {
        return String.format("%s:%s:%s->%s", id, getName(), Arrays.toString(iindices), Arrays.toString(oindices));
    }
}
