package ai.h2o.mojos.runtime.api.backend;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;

/**
 * Utilities for working with {@link ReaderBackend} contents.
 *
 * Note: in Java8, some of these will be best suited as default interface methods in {@link ReaderBackend}.
 */
public class ReaderBackendUtils {

    public static final int BUFFER_SIZE = 16 * 1024;

    // not instantiable
    private ReaderBackendUtils() {}

    /**
     * @return inputstream wrapped as reader, for text processing
     */
    public static BufferedReader asReader(InputStream is) throws IOException {
        return new BufferedReader(new InputStreamReader(is));
    }

    /**
     * Read whole stream and return its bytes
     * @param is -
     * @return bytes from the stream
     * @throws IOException -
     */
    public static byte[] getBytes(InputStream is) throws IOException {
        try (final ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
            copy(is, baos, BUFFER_SIZE);
            return baos.toByteArray();
        }
    }

    /**
     * Extracts subtree of the storage to filesystem.
     * @param backend -
     * @param subdir  the root directory (of the contained files) to be exported
     * @param dest  where to place the exported files
     */
    public static void export(ReaderBackend backend, String subdir, File dest) throws IOException {
        final String prefix = subdir + "/";
        for (String name : backend.list()) {
            if (!name.startsWith(prefix)) continue;
            final File file = new File(dest, name);
            file.getParentFile().mkdirs();
            try (final OutputStream os = new FileOutputStream(file);
                 final InputStream is = backend.getInputStream(name))
            {
                copy(is, os, BUFFER_SIZE);
            }
        }
    }

    /**
     * This is copied from <code>org.codehaus.plexus.util.copy(final InputStream input, final OutputStream output)</code>.
     * Avoiding additional dependency for just one simple method.
     *
     * Copy bytes from an <code>InputStream</code> to an <code>OutputStream</code>.
     *
     * @param bufferSize Size of internal buffer to use.
     */
    public/*?*/ static void copy(final InputStream input, final OutputStream output, final int bufferSize )
            throws IOException
    {
        final byte[] buffer = new byte[bufferSize];
        int n = 0;
        while ( 0 <= ( n = input.read( buffer ) ) )
        {
            output.write( buffer, 0, n );
        }
    }
}
