package ai.h2o.mojos.runtime.api;

/**
 * Immutable configuration of the mojo pipeline.
 */
public class PipelineConfig {
    boolean shapEnabled;
    boolean shapOriginal;
    BasePipelineListener listener = BasePipelineListener.NOOP;
    boolean withPredictionInterval;
    String[] exposedColumnNames;

    public static final PipelineConfig DEFAULT;

    static {
        DEFAULT = PipelineConfig.builder()
            .build()
        ;
    }

    public static PipelineConfig.Builder builder() {
        return new PipelineConfig.Builder();
    }

    private PipelineConfig() {
        // create in building mode
    }

    public boolean isShapEnabled() {
        return shapEnabled;
    }

    public boolean isShapOriginal() {
        return shapOriginal;
    }

    public BasePipelineListener getListener() {
        return listener;
    }

    public boolean isWithPredictionInterval() {
        return withPredictionInterval;
    }

    public String[] getExposedColumnNames() {
        return exposedColumnNames;
    }

    public static class Builder extends PipelineConfig {
        public PipelineConfig build() {
            if (shapOriginal) {
                shapEnabled = true;
            }
            return this;
        }

        /**
         * Allows prediction contribution columns (computed by SHAP algo) to appear in the transformation result.
         * Current implementation removes the standard transformation result. This will be improved in future.
         *
         * @param shapEnabled - currently required to be true.
         * @see #enableShapOriginal(boolean)
         */
        public PipelineConfig.Builder enableShap(final boolean shapEnabled) {
            this.shapEnabled = shapEnabled;
            return this;
        }

        /**
         * If true, switches the pipeline to produce so called "original" SHAP contributions,
         * ie. an estimation how does each pipeline's output contribute to changes in each pipeline's input.
         * And also bias per each output.
         *
         * @param shapOriginal -
         * @see #enableShap(boolean)
         */
        public PipelineConfig.Builder enableShapOriginal(final boolean shapOriginal) {
            if (shapOriginal) {
                enableShap(true);
            }
            this.shapOriginal = shapOriginal;
            return this;
        }


        /**
         * Configures a listener that can report the progress of computation.
         * Useful primarily for debugging and trouble-shooting purposes.
         * <p>
         * Only one listener at a time is currently supported.
         */
        public PipelineConfig.Builder listener(final BasePipelineListener listener) {
            this.listener = listener;
            return this;
        }

        /**
         * Enables exposing prediction interval per each output value.
         */
        public PipelineConfig.Builder withPredictionInterval(final boolean withPredictionInterval) {
            this.withPredictionInterval = withPredictionInterval;
            return this;
        }

        /**
         * @param exposedColumnNames column names that should be part of output frame, no matter if they participate in computation graph.
         */
        public PipelineConfig.Builder exposedColumnNames(final String... exposedColumnNames) {
            this.exposedColumnNames = exposedColumnNames;
            return this;
        }
    }
}
