/*
* Licensed to the Apache Software Foundation (ASF) under one or more
* contributor license agreements.  See the NOTICE file distributed with
* this work for additional information regarding copyright ownership.
* The ASF licenses this file to You under the Apache License, Version 2.0
* (the "License"); you may not use this file except in compliance with
* the License.  You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package ai.h2o.sparkling.ml.params

import ai.h2o.sparkling.ml.algos.H2OGridSearch
import hex.Model
import hex.deeplearning.DeepLearningModel.DeepLearningParameters
import hex.glm.GLMModel.GLMParameters
import hex.tree.gbm.GBMModel.GBMParameters
import hex.tree.xgboost.XGBoostModel.XGBoostParameters
import org.apache.spark.ml.param.{Param, Params}
import org.json4s.JsonAST.{JArray, JInt}
import org.json4s.jackson.JsonMethods.{compact, parse, render}
import org.json4s.{JNull, JValue}
import water.AutoBuffer


class AlgoParams(parent: Params, name: String, doc: String, isValid: Model.Parameters => Boolean)
  extends Param[Model.Parameters](parent, name, doc, isValid) {

  def this(parent: Params, name: String, doc: String) =
    this(parent, name, doc, _ => true)

  override def jsonEncode(value: Model.Parameters): String = {
    val encoded: JValue = if (value == null) {
      JNull
    } else {
      val ab = new AutoBuffer()
      ab.putStr(value.algoName())
      value.write(ab)
      val bytes = ab.buf()
      JArray(bytes.toSeq.map(JInt(_)).toList)
    }
    compact(render(encoded))
  }

  override def jsonDecode(json: String): Model.Parameters = {
    parse(json) match {
      case JNull =>
        null.asInstanceOf[Model.Parameters]
      case JArray(values) =>
        val bytes = values.map {
          case JInt(x) =>
            x.byteValue()
          case _ =>
            throw new IllegalArgumentException(s"Cannot decode $json to Byte.")
        }.toArray
        val ab = new AutoBuffer(bytes)
        val algoName = ab.getStr
        val params = H2OGridSearch.SupportedAlgos.fromString(algoName).get  match {
          case H2OGridSearch.SupportedAlgos.glm => new GLMParameters()
          case H2OGridSearch.SupportedAlgos.gbm => new GBMParameters()
          case H2OGridSearch.SupportedAlgos.deeplearning => new DeepLearningParameters()
          case H2OGridSearch.SupportedAlgos.xgboost => new XGBoostParameters()

          case _ => throw new RuntimeException("Not supported algorithm")

        }
        params.read(ab)
        params.asInstanceOf[Model.Parameters]
      case _ =>
        throw new IllegalArgumentException(s"Cannot decode $json to a class for desired algo.")
    }
  }
}
