/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ai.h2o.sparkling.ml.params

import ai.h2o.automl.AutoMLBuildSpec.AutoMLBuildModels
import ai.h2o.sparkling.H2OFrame

trait H2OAutoMLBuildModelsParams
  extends H2OAlgoParamsBase {


  //
  // Parameter definitions
  //
  protected val excludeAlgos = nullableStringArrayParam(
    name = "excludeAlgos",
    doc = """A list of algorithms to skip during the model-building phase. Possible values are ``"GLM"``, ``"DRF"``, ``"GBM"``, ``"DeepLearning"``, ``"StackedEnsemble"``, ``"XGBoost"``.""")

  protected val includeAlgos = stringArrayParam(
    name = "includeAlgos",
    doc = """A list of algorithms to restrict to during the model-building phase. Possible values are ``"GLM"``, ``"DRF"``, ``"GBM"``, ``"DeepLearning"``, ``"StackedEnsemble"``, ``"XGBoost"``.""")

  protected val exploitationRatio = doubleParam(
    name = "exploitationRatio",
    doc = """The budget ratio (between 0 and 1) dedicated to the exploitation (vs exploration) phase.""")

  //
  // Default values
  //
  setDefault(
    excludeAlgos -> null,
    includeAlgos -> Array("GLM", "DRF", "GBM", "DeepLearning", "StackedEnsemble", "XGBoost"),
    exploitationRatio -> 0.0)

  //
  // Getters
  //
  def getExcludeAlgos(): Array[String] = $(excludeAlgos)

  def getIncludeAlgos(): Array[String] = $(includeAlgos)

  def getExploitationRatio(): Double = $(exploitationRatio)

  //
  // Setters
  //
  def setExcludeAlgos(value: Array[String]): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValues[ai.h2o.automl.Algo](value, nullEnabled = true)
    set(excludeAlgos, validated)
  }
           
  def setIncludeAlgos(value: Array[String]): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValues[ai.h2o.automl.Algo](value, nullEnabled = true)
    set(includeAlgos, validated)
  }
           
  def setExploitationRatio(value: Double): this.type = {
    set(exploitationRatio, value)
  }
           

  override private[sparkling] def getH2OAlgorithmParams(trainingFrame: H2OFrame): Map[String, Any] = {
    super.getH2OAlgorithmParams(trainingFrame) ++ getH2OAutoMLBuildModelsParams()
  }

  private[sparkling] def getH2OAutoMLBuildModelsParams(): Map[String, Any] = {
      Map(
        "exclude_algos" -> getExcludeAlgos(),
        "include_algos" -> getIncludeAlgos(),
        "exploitation_ratio" -> getExploitationRatio())
  }

  override private[sparkling] def getSWtoH2OParamNameMap(): Map[String, String] = {
    super.getSWtoH2OParamNameMap() ++
      Map(
        "excludeAlgos" -> "exclude_algos",
        "includeAlgos" -> "include_algos",
        "exploitationRatio" -> "exploitation_ratio")
  }
      
}
    