/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ai.h2o.sparkling.ml.params

import hex.grid.HyperSpaceSearchCriteria.CartesianSearchCriteria
import ai.h2o.sparkling.H2OFrame
import hex.grid.HyperSpaceSearchCriteria.Strategy

trait H2OGridSearchCommonCriteriaParams
  extends H2OAlgoParamsBase {


  //
  // Parameter definitions
  //
  protected val strategy = stringParam(
    name = "strategy",
    doc = """Hyperparameter space search strategy. Possible values are ``"Unknown"``, ``"Cartesian"``, ``"RandomDiscrete"``, ``"Sequential"``.""")

  //
  // Default values
  //
  setDefault(
    strategy -> Strategy.Cartesian.name())

  //
  // Getters
  //
  def getStrategy(): String = $(strategy)

  //
  // Setters
  //
  def setStrategy(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[Strategy](value)
    set(strategy, validated)
  }
           

  override private[sparkling] def getH2OAlgorithmParams(trainingFrame: H2OFrame): Map[String, Any] = {
    super.getH2OAlgorithmParams(trainingFrame) ++ getH2OGridSearchCommonCriteriaParams(trainingFrame)
  }

  private[sparkling] def getH2OGridSearchCommonCriteriaParams(trainingFrame: H2OFrame): Map[String, Any] = {
      Map(
        "strategy" -> getStrategy())
  }

  override private[sparkling] def getSWtoH2OParamNameMap(): Map[String, String] = {
    super.getSWtoH2OParamNameMap() ++
      Map(
        "strategy" -> "strategy")
  }
      
}
