/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ai.h2o.sparkling.ml.params

import hex.gam.GAMModel.GAMParameters
import ai.h2o.sparkling.H2OFrame
import hex.glm.GLMModel.GLMParameters.Family
import hex.glm.GLMModel.GLMParameters.Solver
import hex.glm.GLMModel.GLMParameters.MissingValuesHandling
import hex.glm.GLMModel.GLMParameters.Link
import hex.Model.Parameters.FoldAssignmentScheme
import hex.ScoreKeeper.StoppingMetric
import hex.MultinomialAucType

trait H2OGAMParams
  extends H2OAlgoParamsBase
  with HasIgnoredCols
  with HasBetaConstraints
  with HasGamCols {

  protected def paramTag = reflect.classTag[GAMParameters]

  //
  // Parameter definitions
  //
  protected val seed = longParam(
    name = "seed",
    doc = """Seed for pseudo random number generator (if applicable).""")

  protected val family = stringParam(
    name = "family",
    doc = """Family. Use binomial for classification with logistic regression, others are for regression problems. Possible values are ``"AUTO"``, ``"gaussian"``, ``"binomial"``, ``"fractionalbinomial"``, ``"quasibinomial"``, ``"poisson"``, ``"gamma"``, ``"multinomial"``, ``"tweedie"``, ``"ordinal"``, ``"negativebinomial"``.""")

  protected val tweedieVariancePower = doubleParam(
    name = "tweedieVariancePower",
    doc = """Tweedie variance power.""")

  protected val tweedieLinkPower = doubleParam(
    name = "tweedieLinkPower",
    doc = """Tweedie link power.""")

  protected val theta = doubleParam(
    name = "theta",
    doc = """Theta.""")

  protected val solver = stringParam(
    name = "solver",
    doc = """AUTO will set the solver based on given data and the other parameters. IRLSM is fast on on problems with small number of predictors and for lambda-search with L1 penalty, L_BFGS scales better for datasets with many columns. Possible values are ``"AUTO"``, ``"IRLSM"``, ``"L_BFGS"``, ``"COORDINATE_DESCENT_NAIVE"``, ``"COORDINATE_DESCENT"``, ``"GRADIENT_DESCENT_LH"``, ``"GRADIENT_DESCENT_SQERR"``.""")

  protected val alphaValue = nullableDoubleArrayParam(
    name = "alphaValue",
    doc = """Distribution of regularization between the L1 (Lasso) and L2 (Ridge) penalties. A value of 1 for alpha represents Lasso regression, a value of 0 produces Ridge regression, and anything in between specifies the amount of mixing between the two. Default value of alpha is 0 when SOLVER = 'L-BFGS'; 0.5 otherwise.""")

  protected val lambdaValue = nullableDoubleArrayParam(
    name = "lambdaValue",
    doc = """Regularization strength.""")

  protected val startval = nullableDoubleArrayParam(
    name = "startval",
    doc = """double array to initialize coefficients for GAM.""")

  protected val lambdaSearch = booleanParam(
    name = "lambdaSearch",
    doc = """Use lambda search starting at lambda max, given lambda is then interpreted as lambda min.""")

  protected val earlyStopping = booleanParam(
    name = "earlyStopping",
    doc = """Stop early when there is no more relative improvement on train or validation (if provided).""")

  protected val nlambdas = intParam(
    name = "nlambdas",
    doc = """Number of lambdas to be used in a search. Default indicates: If alpha is zero, with lambda search set to True, the value of nlamdas is set to 30 (fewer lambdas are needed for ridge regression) otherwise it is set to 100.""")

  protected val standardize = booleanParam(
    name = "standardize",
    doc = """Standardize numeric columns to have zero mean and unit variance.""")

  protected val missingValuesHandling = stringParam(
    name = "missingValuesHandling",
    doc = """Handling of missing values. Either MeanImputation, Skip or PlugValues. Possible values are ``"MeanImputation"``, ``"PlugValues"``, ``"Skip"``.""")

  protected val nonNegative = booleanParam(
    name = "nonNegative",
    doc = """Restrict coefficients (not intercept) to be non-negative.""")

  protected val maxIterations = intParam(
    name = "maxIterations",
    doc = """Maximum number of iterations.""")

  protected val betaEpsilon = doubleParam(
    name = "betaEpsilon",
    doc = """Converge if  beta changes less (using L-infinity norm) than beta esilon, ONLY applies to IRLSM solver .""")

  protected val objectiveEpsilon = doubleParam(
    name = "objectiveEpsilon",
    doc = """Converge if  objective value changes less than this. Default indicates: If lambda_search is set to True the value of objective_epsilon is set to .0001. If the lambda_search is set to False and lambda is equal to zero, the value of objective_epsilon is set to .000001, for any other value of lambda the default value of objective_epsilon is set to .0001.""")

  protected val gradientEpsilon = doubleParam(
    name = "gradientEpsilon",
    doc = """Converge if  objective changes less (using L-infinity norm) than this, ONLY applies to L-BFGS solver. Default indicates: If lambda_search is set to False and lambda is equal to zero, the default value of gradient_epsilon is equal to .000001, otherwise the default value is .0001. If lambda_search is set to True, the conditional values above are 1E-8 and 1E-6 respectively.""")

  protected val objReg = doubleParam(
    name = "objReg",
    doc = """Likelihood divider in objective value computation, default is 1/nobs.""")

  protected val link = stringParam(
    name = "link",
    doc = """Link function. Possible values are ``"family_default"``, ``"identity"``, ``"logit"``, ``"log"``, ``"inverse"``, ``"tweedie"``, ``"multinomial"``, ``"ologit"``, ``"oprobit"``, ``"ologlog"``.""")

  protected val intercept = booleanParam(
    name = "intercept",
    doc = """Include constant term in the model.""")

  protected val prior = doubleParam(
    name = "prior",
    doc = """Prior probability for y==1. To be used only for logistic regression iff the data has been sampled and the mean of response does not reflect reality.""")

  protected val coldStart = booleanParam(
    name = "coldStart",
    doc = """Only applicable to multiple alpha/lambda values when calling GLM from GAM.  If false, build the next model for next set of alpha/lambda values starting from the values provided by current model.  If true will start GLM model from scratch.""")

  protected val maxActivePredictors = intParam(
    name = "maxActivePredictors",
    doc = """Maximum number of active predictors during computation. Use as a stopping criterion to prevent expensive model building with many predictors. Default indicates: If the IRLSM solver is used, the value of max_active_predictors is set to 5000 otherwise it is set to 100000000.""")

  protected val interactions = nullableStringArrayParam(
    name = "interactions",
    doc = """A list of predictor column indices to interact. All pairwise combinations will be computed for the list.""")

  protected val balanceClasses = booleanParam(
    name = "balanceClasses",
    doc = """Balance training data class counts via over/under-sampling (for imbalanced data).""")

  protected val classSamplingFactors = nullableFloatArrayParam(
    name = "classSamplingFactors",
    doc = """Desired over/under-sampling ratios per class (in lexicographic order). If not specified, sampling factors will be automatically computed to obtain class balance during training. Requires balance_classes.""")

  protected val maxAfterBalanceSize = floatParam(
    name = "maxAfterBalanceSize",
    doc = """Maximum relative size of the training data after balancing class counts (can be less than 1.0). Requires balance_classes.""")

  protected val maxConfusionMatrixSize = intParam(
    name = "maxConfusionMatrixSize",
    doc = """[Deprecated] Maximum size (# classes) for confusion matrices to be printed in the Logs.""")

  protected val computePValues = booleanParam(
    name = "computePValues",
    doc = """Request p-values computation, p-values work only with IRLSM solver and no regularization.""")

  protected val removeCollinearCols = booleanParam(
    name = "removeCollinearCols",
    doc = """In case of linearly dependent columns, remove some of the dependent columns.""")

  protected val numKnots = nullableIntArrayParam(
    name = "numKnots",
    doc = """Number of knots for gam predictors.""")

  protected val scale = nullableDoubleArrayParam(
    name = "scale",
    doc = """Smoothing parameter for gam predictors.  If specified, must be of the same length as gam_columns.""")

  protected val bs = nullableIntArrayParam(
    name = "bs",
    doc = """Basis function type for each gam predictors, 0 for cr, 1 for thin plate regression with knots, 2 for thin plate regression with SVD.  If specified, must be the same size as gam_columns.""")

  protected val keepGamCols = booleanParam(
    name = "keepGamCols",
    doc = """Save keys of model matrix.""")

  protected val standardizeTpGamCols = booleanParam(
    name = "standardizeTpGamCols",
    doc = """standardize tp (thin plate) predictor columns.""")

  protected val scaleTpPenaltyMat = booleanParam(
    name = "scaleTpPenaltyMat",
    doc = """Scale penalty matrix for tp (thin plate) smoothers as in R.""")

  protected val knotIds = nullableStringArrayParam(
    name = "knotIds",
    doc = """String arrays storing frame keys of knots.  One for each gam column set specified in gam_columns.""")

  protected val modelId = nullableStringParam(
    name = "modelId",
    doc = """Destination id for this model; auto-generated if not specified.""")

  protected val nfolds = intParam(
    name = "nfolds",
    doc = """Number of folds for K-fold cross-validation (0 to disable or >= 2).""")

  protected val keepCrossValidationModels = booleanParam(
    name = "keepCrossValidationModels",
    doc = """Whether to keep the cross-validation models.""")

  protected val keepCrossValidationPredictions = booleanParam(
    name = "keepCrossValidationPredictions",
    doc = """Whether to keep the predictions of the cross-validation models.""")

  protected val keepCrossValidationFoldAssignment = booleanParam(
    name = "keepCrossValidationFoldAssignment",
    doc = """Whether to keep the cross-validation fold assignment.""")

  protected val labelCol = stringParam(
    name = "labelCol",
    doc = """Response variable column.""")

  protected val weightCol = nullableStringParam(
    name = "weightCol",
    doc = """Column with observation weights. Giving some observation a weight of zero is equivalent to excluding it from the dataset; giving an observation a relative weight of 2 is equivalent to repeating that row twice. Negative weights are not allowed. Note: Weights are per-row observation weights and do not increase the size of the data frame. This is typically the number of times a row is repeated, but non-integer values are supported as well. During training, rows with higher weights matter more, due to the larger loss function pre-factor.""")

  protected val offsetCol = nullableStringParam(
    name = "offsetCol",
    doc = """Offset column. This will be added to the combination of columns before applying the link function.""")

  protected val foldCol = nullableStringParam(
    name = "foldCol",
    doc = """Column with cross-validation fold index assignment per observation.""")

  protected val foldAssignment = stringParam(
    name = "foldAssignment",
    doc = """Cross-validation fold assignment scheme, if fold_column is not specified. The 'Stratified' option will stratify the folds based on the response variable, for classification problems. Possible values are ``"AUTO"``, ``"Random"``, ``"Modulo"``, ``"Stratified"``.""")

  protected val ignoreConstCols = booleanParam(
    name = "ignoreConstCols",
    doc = """Ignore constant columns.""")

  protected val scoreEachIteration = booleanParam(
    name = "scoreEachIteration",
    doc = """Whether to score during each iteration of model training.""")

  protected val stoppingRounds = intParam(
    name = "stoppingRounds",
    doc = """Early stopping based on convergence of stopping_metric. Stop if simple moving average of length k of the stopping_metric does not improve for k:=stopping_rounds scoring events (0 to disable).""")

  protected val maxRuntimeSecs = doubleParam(
    name = "maxRuntimeSecs",
    doc = """Maximum allowed runtime in seconds for model training. Use 0 to disable.""")

  protected val stoppingMetric = stringParam(
    name = "stoppingMetric",
    doc = """Metric to use for early stopping (AUTO: logloss for classification, deviance for regression and anonomaly_score for Isolation Forest). Note that custom and custom_increasing can only be used in GBM and DRF with the Python client. Possible values are ``"AUTO"``, ``"deviance"``, ``"logloss"``, ``"MSE"``, ``"RMSE"``, ``"MAE"``, ``"RMSLE"``, ``"AUC"``, ``"AUCPR"``, ``"lift_top_group"``, ``"misclassification"``, ``"mean_per_class_error"``, ``"anomaly_score"``, ``"custom"``, ``"custom_increasing"``.""")

  protected val stoppingTolerance = doubleParam(
    name = "stoppingTolerance",
    doc = """Relative tolerance for metric-based stopping criterion (stop if relative improvement is not at least this much).""")

  protected val customMetricFunc = nullableStringParam(
    name = "customMetricFunc",
    doc = """Reference to custom evaluation function, format: `language:keyName=funcName`.""")

  protected val exportCheckpointsDir = nullableStringParam(
    name = "exportCheckpointsDir",
    doc = """Automatically export generated models to this directory.""")

  protected val aucType = stringParam(
    name = "aucType",
    doc = """Set default multinomial AUC type. Possible values are ``"AUTO"``, ``"NONE"``, ``"MACRO_OVR"``, ``"WEIGHTED_OVR"``, ``"MACRO_OVO"``, ``"WEIGHTED_OVO"``.""")

  //
  // Default values
  //
  setDefault(
    seed -> -1L,
    family -> Family.AUTO.name(),
    tweedieVariancePower -> 0.0,
    tweedieLinkPower -> 0.0,
    theta -> 0.0,
    solver -> Solver.AUTO.name(),
    alphaValue -> null,
    lambdaValue -> null,
    startval -> null,
    lambdaSearch -> false,
    earlyStopping -> true,
    nlambdas -> -1,
    standardize -> false,
    missingValuesHandling -> MissingValuesHandling.MeanImputation.name(),
    nonNegative -> false,
    maxIterations -> -1,
    betaEpsilon -> 1.0e-4,
    objectiveEpsilon -> -1.0,
    gradientEpsilon -> -1.0,
    objReg -> -1.0,
    link -> Link.family_default.name(),
    intercept -> true,
    prior -> -1.0,
    coldStart -> false,
    maxActivePredictors -> -1,
    interactions -> null,
    balanceClasses -> false,
    classSamplingFactors -> null,
    maxAfterBalanceSize -> 5.0f,
    maxConfusionMatrixSize -> 20,
    computePValues -> false,
    removeCollinearCols -> false,
    numKnots -> null,
    scale -> null,
    bs -> null,
    keepGamCols -> false,
    standardizeTpGamCols -> false,
    scaleTpPenaltyMat -> false,
    knotIds -> null,
    modelId -> null,
    nfolds -> 0,
    keepCrossValidationModels -> true,
    keepCrossValidationPredictions -> false,
    keepCrossValidationFoldAssignment -> false,
    labelCol -> "label",
    weightCol -> null,
    offsetCol -> null,
    foldCol -> null,
    foldAssignment -> FoldAssignmentScheme.AUTO.name(),
    ignoreConstCols -> true,
    scoreEachIteration -> false,
    stoppingRounds -> 0,
    maxRuntimeSecs -> 0.0,
    stoppingMetric -> StoppingMetric.AUTO.name(),
    stoppingTolerance -> 0.001,
    customMetricFunc -> null,
    exportCheckpointsDir -> null,
    aucType -> MultinomialAucType.AUTO.name())

  //
  // Getters
  //
  def getSeed(): Long = $(seed)

  def getFamily(): String = $(family)

  def getTweedieVariancePower(): Double = $(tweedieVariancePower)

  def getTweedieLinkPower(): Double = $(tweedieLinkPower)

  def getTheta(): Double = $(theta)

  def getSolver(): String = $(solver)

  def getAlphaValue(): Array[Double] = $(alphaValue)

  def getLambdaValue(): Array[Double] = $(lambdaValue)

  def getStartval(): Array[Double] = $(startval)

  def getLambdaSearch(): Boolean = $(lambdaSearch)

  def getEarlyStopping(): Boolean = $(earlyStopping)

  def getNlambdas(): Int = $(nlambdas)

  def getStandardize(): Boolean = $(standardize)

  def getMissingValuesHandling(): String = $(missingValuesHandling)

  def getNonNegative(): Boolean = $(nonNegative)

  def getMaxIterations(): Int = $(maxIterations)

  def getBetaEpsilon(): Double = $(betaEpsilon)

  def getObjectiveEpsilon(): Double = $(objectiveEpsilon)

  def getGradientEpsilon(): Double = $(gradientEpsilon)

  def getObjReg(): Double = $(objReg)

  def getLink(): String = $(link)

  def getIntercept(): Boolean = $(intercept)

  def getPrior(): Double = $(prior)

  def getColdStart(): Boolean = $(coldStart)

  def getMaxActivePredictors(): Int = $(maxActivePredictors)

  def getInteractions(): Array[String] = $(interactions)

  def getBalanceClasses(): Boolean = $(balanceClasses)

  def getClassSamplingFactors(): Array[Float] = $(classSamplingFactors)

  def getMaxAfterBalanceSize(): Float = $(maxAfterBalanceSize)

  def getMaxConfusionMatrixSize(): Int = $(maxConfusionMatrixSize)

  def getComputePValues(): Boolean = $(computePValues)

  def getRemoveCollinearCols(): Boolean = $(removeCollinearCols)

  def getNumKnots(): Array[Int] = $(numKnots)

  def getScale(): Array[Double] = $(scale)

  def getBs(): Array[Int] = $(bs)

  def getKeepGamCols(): Boolean = $(keepGamCols)

  def getStandardizeTpGamCols(): Boolean = $(standardizeTpGamCols)

  def getScaleTpPenaltyMat(): Boolean = $(scaleTpPenaltyMat)

  def getKnotIds(): Array[String] = $(knotIds)

  def getModelId(): String = $(modelId)

  def getNfolds(): Int = $(nfolds)

  def getKeepCrossValidationModels(): Boolean = $(keepCrossValidationModels)

  def getKeepCrossValidationPredictions(): Boolean = $(keepCrossValidationPredictions)

  def getKeepCrossValidationFoldAssignment(): Boolean = $(keepCrossValidationFoldAssignment)

  def getLabelCol(): String = $(labelCol)

  def getWeightCol(): String = $(weightCol)

  def getOffsetCol(): String = $(offsetCol)

  def getFoldCol(): String = $(foldCol)

  def getFoldAssignment(): String = $(foldAssignment)

  def getIgnoreConstCols(): Boolean = $(ignoreConstCols)

  def getScoreEachIteration(): Boolean = $(scoreEachIteration)

  def getStoppingRounds(): Int = $(stoppingRounds)

  def getMaxRuntimeSecs(): Double = $(maxRuntimeSecs)

  def getStoppingMetric(): String = $(stoppingMetric)

  def getStoppingTolerance(): Double = $(stoppingTolerance)

  def getCustomMetricFunc(): String = $(customMetricFunc)

  def getExportCheckpointsDir(): String = $(exportCheckpointsDir)

  def getAucType(): String = $(aucType)

  //
  // Setters
  //
  def setSeed(value: Long): this.type = {
    set(seed, value)
  }
           
  def setFamily(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[Family](value)
    set(family, validated)
  }
           
  def setTweedieVariancePower(value: Double): this.type = {
    set(tweedieVariancePower, value)
  }
           
  def setTweedieLinkPower(value: Double): this.type = {
    set(tweedieLinkPower, value)
  }
           
  def setTheta(value: Double): this.type = {
    set(theta, value)
  }
           
  def setSolver(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[Solver](value)
    set(solver, validated)
  }
           
  def setAlphaValue(value: Array[Double]): this.type = {
    set(alphaValue, value)
  }
           
  def setLambdaValue(value: Array[Double]): this.type = {
    set(lambdaValue, value)
  }
           
  def setStartval(value: Array[Double]): this.type = {
    set(startval, value)
  }
           
  def setLambdaSearch(value: Boolean): this.type = {
    set(lambdaSearch, value)
  }
           
  def setEarlyStopping(value: Boolean): this.type = {
    set(earlyStopping, value)
  }
           
  def setNlambdas(value: Int): this.type = {
    set(nlambdas, value)
  }
           
  def setStandardize(value: Boolean): this.type = {
    set(standardize, value)
  }
           
  def setMissingValuesHandling(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[MissingValuesHandling](value)
    set(missingValuesHandling, validated)
  }
           
  def setNonNegative(value: Boolean): this.type = {
    set(nonNegative, value)
  }
           
  def setMaxIterations(value: Int): this.type = {
    set(maxIterations, value)
  }
           
  def setBetaEpsilon(value: Double): this.type = {
    set(betaEpsilon, value)
  }
           
  def setObjectiveEpsilon(value: Double): this.type = {
    set(objectiveEpsilon, value)
  }
           
  def setGradientEpsilon(value: Double): this.type = {
    set(gradientEpsilon, value)
  }
           
  def setObjReg(value: Double): this.type = {
    set(objReg, value)
  }
           
  def setLink(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[Link](value)
    set(link, validated)
  }
           
  def setIntercept(value: Boolean): this.type = {
    set(intercept, value)
  }
           
  def setPrior(value: Double): this.type = {
    set(prior, value)
  }
           
  def setColdStart(value: Boolean): this.type = {
    set(coldStart, value)
  }
           
  def setMaxActivePredictors(value: Int): this.type = {
    set(maxActivePredictors, value)
  }
           
  def setInteractions(value: Array[String]): this.type = {
    set(interactions, value)
  }
           
  def setBalanceClasses(value: Boolean): this.type = {
    set(balanceClasses, value)
  }
           
  def setClassSamplingFactors(value: Array[Float]): this.type = {
    set(classSamplingFactors, value)
  }
           
  def setMaxAfterBalanceSize(value: Float): this.type = {
    set(maxAfterBalanceSize, value)
  }
           
  def setMaxConfusionMatrixSize(value: Int): this.type = {
    set(maxConfusionMatrixSize, value)
  }
           
  def setComputePValues(value: Boolean): this.type = {
    set(computePValues, value)
  }
           
  def setRemoveCollinearCols(value: Boolean): this.type = {
    set(removeCollinearCols, value)
  }
           
  def setNumKnots(value: Array[Int]): this.type = {
    set(numKnots, value)
  }
           
  def setScale(value: Array[Double]): this.type = {
    set(scale, value)
  }
           
  def setBs(value: Array[Int]): this.type = {
    set(bs, value)
  }
           
  def setKeepGamCols(value: Boolean): this.type = {
    set(keepGamCols, value)
  }
           
  def setStandardizeTpGamCols(value: Boolean): this.type = {
    set(standardizeTpGamCols, value)
  }
           
  def setScaleTpPenaltyMat(value: Boolean): this.type = {
    set(scaleTpPenaltyMat, value)
  }
           
  def setKnotIds(value: Array[String]): this.type = {
    set(knotIds, value)
  }
           
  def setModelId(value: String): this.type = {
    set(modelId, value)
  }
           
  def setNfolds(value: Int): this.type = {
    set(nfolds, value)
  }
           
  def setKeepCrossValidationModels(value: Boolean): this.type = {
    set(keepCrossValidationModels, value)
  }
           
  def setKeepCrossValidationPredictions(value: Boolean): this.type = {
    set(keepCrossValidationPredictions, value)
  }
           
  def setKeepCrossValidationFoldAssignment(value: Boolean): this.type = {
    set(keepCrossValidationFoldAssignment, value)
  }
           
  def setLabelCol(value: String): this.type = {
    set(labelCol, value)
  }
           
  def setWeightCol(value: String): this.type = {
    set(weightCol, value)
  }
           
  def setOffsetCol(value: String): this.type = {
    set(offsetCol, value)
  }
           
  def setFoldCol(value: String): this.type = {
    set(foldCol, value)
  }
           
  def setFoldAssignment(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[FoldAssignmentScheme](value)
    set(foldAssignment, validated)
  }
           
  def setIgnoreConstCols(value: Boolean): this.type = {
    set(ignoreConstCols, value)
  }
           
  def setScoreEachIteration(value: Boolean): this.type = {
    set(scoreEachIteration, value)
  }
           
  def setStoppingRounds(value: Int): this.type = {
    set(stoppingRounds, value)
  }
           
  def setMaxRuntimeSecs(value: Double): this.type = {
    set(maxRuntimeSecs, value)
  }
           
  def setStoppingMetric(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[StoppingMetric](value)
    set(stoppingMetric, validated)
  }
           
  def setStoppingTolerance(value: Double): this.type = {
    set(stoppingTolerance, value)
  }
           
  def setCustomMetricFunc(value: String): this.type = {
    set(customMetricFunc, value)
  }
           
  def setExportCheckpointsDir(value: String): this.type = {
    set(exportCheckpointsDir, value)
  }
           
  def setAucType(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[MultinomialAucType](value)
    set(aucType, validated)
  }
           

  override private[sparkling] def getH2OAlgorithmParams(trainingFrame: H2OFrame): Map[String, Any] = {
    super.getH2OAlgorithmParams(trainingFrame) ++ getH2OGAMParams(trainingFrame)
  }

  private[sparkling] def getH2OGAMParams(trainingFrame: H2OFrame): Map[String, Any] = {
      Map(
        "seed" -> getSeed(),
        "family" -> getFamily(),
        "tweedie_variance_power" -> getTweedieVariancePower(),
        "tweedie_link_power" -> getTweedieLinkPower(),
        "theta" -> getTheta(),
        "solver" -> getSolver(),
        "alpha" -> getAlphaValue(),
        "lambda" -> getLambdaValue(),
        "startval" -> getStartval(),
        "lambda_search" -> getLambdaSearch(),
        "early_stopping" -> getEarlyStopping(),
        "nlambdas" -> getNlambdas(),
        "standardize" -> getStandardize(),
        "missing_values_handling" -> getMissingValuesHandling(),
        "non_negative" -> getNonNegative(),
        "max_iterations" -> getMaxIterations(),
        "beta_epsilon" -> getBetaEpsilon(),
        "objective_epsilon" -> getObjectiveEpsilon(),
        "gradient_epsilon" -> getGradientEpsilon(),
        "obj_reg" -> getObjReg(),
        "link" -> getLink(),
        "intercept" -> getIntercept(),
        "prior" -> getPrior(),
        "cold_start" -> getColdStart(),
        "max_active_predictors" -> getMaxActivePredictors(),
        "interactions" -> getInteractions(),
        "balance_classes" -> getBalanceClasses(),
        "class_sampling_factors" -> getClassSamplingFactors(),
        "max_after_balance_size" -> getMaxAfterBalanceSize(),
        "max_confusion_matrix_size" -> getMaxConfusionMatrixSize(),
        "compute_p_values" -> getComputePValues(),
        "remove_collinear_columns" -> getRemoveCollinearCols(),
        "num_knots" -> getNumKnots(),
        "scale" -> getScale(),
        "bs" -> getBs(),
        "keep_gam_cols" -> getKeepGamCols(),
        "standardize_tp_gam_cols" -> getStandardizeTpGamCols(),
        "scale_tp_penalty_mat" -> getScaleTpPenaltyMat(),
        "knot_ids" -> getKnotIds(),
        "model_id" -> getModelId(),
        "nfolds" -> getNfolds(),
        "keep_cross_validation_models" -> getKeepCrossValidationModels(),
        "keep_cross_validation_predictions" -> getKeepCrossValidationPredictions(),
        "keep_cross_validation_fold_assignment" -> getKeepCrossValidationFoldAssignment(),
        "response_column" -> getLabelCol(),
        "weights_column" -> getWeightCol(),
        "offset_column" -> getOffsetCol(),
        "fold_column" -> getFoldCol(),
        "fold_assignment" -> getFoldAssignment(),
        "ignore_const_cols" -> getIgnoreConstCols(),
        "score_each_iteration" -> getScoreEachIteration(),
        "stopping_rounds" -> getStoppingRounds(),
        "max_runtime_secs" -> getMaxRuntimeSecs(),
        "stopping_metric" -> getStoppingMetric(),
        "stopping_tolerance" -> getStoppingTolerance(),
        "custom_metric_func" -> getCustomMetricFunc(),
        "export_checkpoints_dir" -> getExportCheckpointsDir(),
        "auc_type" -> getAucType()) +++
      getIgnoredColsParam(trainingFrame) +++
      getBetaConstraintsParam(trainingFrame) +++
      getGamColsParam(trainingFrame)
  }

  override private[sparkling] def getSWtoH2OParamNameMap(): Map[String, String] = {
    super.getSWtoH2OParamNameMap() ++
      Map(
        "seed" -> "seed",
        "family" -> "family",
        "tweedieVariancePower" -> "tweedie_variance_power",
        "tweedieLinkPower" -> "tweedie_link_power",
        "theta" -> "theta",
        "solver" -> "solver",
        "alphaValue" -> "alpha",
        "lambdaValue" -> "lambda",
        "startval" -> "startval",
        "lambdaSearch" -> "lambda_search",
        "earlyStopping" -> "early_stopping",
        "nlambdas" -> "nlambdas",
        "standardize" -> "standardize",
        "missingValuesHandling" -> "missing_values_handling",
        "nonNegative" -> "non_negative",
        "maxIterations" -> "max_iterations",
        "betaEpsilon" -> "beta_epsilon",
        "objectiveEpsilon" -> "objective_epsilon",
        "gradientEpsilon" -> "gradient_epsilon",
        "objReg" -> "obj_reg",
        "link" -> "link",
        "intercept" -> "intercept",
        "prior" -> "prior",
        "coldStart" -> "cold_start",
        "maxActivePredictors" -> "max_active_predictors",
        "interactions" -> "interactions",
        "balanceClasses" -> "balance_classes",
        "classSamplingFactors" -> "class_sampling_factors",
        "maxAfterBalanceSize" -> "max_after_balance_size",
        "maxConfusionMatrixSize" -> "max_confusion_matrix_size",
        "computePValues" -> "compute_p_values",
        "removeCollinearCols" -> "remove_collinear_columns",
        "numKnots" -> "num_knots",
        "scale" -> "scale",
        "bs" -> "bs",
        "keepGamCols" -> "keep_gam_cols",
        "standardizeTpGamCols" -> "standardize_tp_gam_cols",
        "scaleTpPenaltyMat" -> "scale_tp_penalty_mat",
        "knotIds" -> "knot_ids",
        "modelId" -> "model_id",
        "nfolds" -> "nfolds",
        "keepCrossValidationModels" -> "keep_cross_validation_models",
        "keepCrossValidationPredictions" -> "keep_cross_validation_predictions",
        "keepCrossValidationFoldAssignment" -> "keep_cross_validation_fold_assignment",
        "labelCol" -> "response_column",
        "weightCol" -> "weights_column",
        "offsetCol" -> "offset_column",
        "foldCol" -> "fold_column",
        "foldAssignment" -> "fold_assignment",
        "ignoreConstCols" -> "ignore_const_cols",
        "scoreEachIteration" -> "score_each_iteration",
        "stoppingRounds" -> "stopping_rounds",
        "maxRuntimeSecs" -> "max_runtime_secs",
        "stoppingMetric" -> "stopping_metric",
        "stoppingTolerance" -> "stopping_tolerance",
        "customMetricFunc" -> "custom_metric_func",
        "exportCheckpointsDir" -> "export_checkpoints_dir",
        "aucType" -> "auc_type")
  }
      
}
    