/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ai.h2o.sparkling.ml.params

import ai.h2o.automl.AutoMLBuildSpec.AutoMLBuildControl
import ai.h2o.sparkling.H2OFrame
import hex.genmodel.utils.DistributionFamily

trait H2OAutoMLBuildControlParams
  extends H2OAlgoParamsBase {


  //
  // Parameter definitions
  //
  protected val projectName = nullableStringParam(
    name = "projectName",
    doc = """Optional project name used to group models from multiple AutoML runs into a single Leaderboard; derived from the training data name if not specified.""")

  protected val nfolds = intParam(
    name = "nfolds",
    doc = """Number of folds for k-fold cross-validation (defaults to -1 (AUTO), otherwise it must be >=2 or use 0 to disable). Disabling prevents Stacked Ensembles from being built.""")

  protected val balanceClasses = booleanParam(
    name = "balanceClasses",
    doc = """Balance training data class counts via over/under-sampling (for imbalanced data).""")

  protected val classSamplingFactors = nullableFloatArrayParam(
    name = "classSamplingFactors",
    doc = """Desired over/under-sampling ratios per class (in lexicographic order). If not specified, sampling factors will be automatically computed to obtain class balance during training. Requires balance_classes.""")

  protected val maxAfterBalanceSize = floatParam(
    name = "maxAfterBalanceSize",
    doc = """Maximum relative size of the training data after balancing class counts (defaults to 5.0 and can be less than 1.0). Requires balance_classes.""")

  protected val keepCrossValidationPredictions = booleanParam(
    name = "keepCrossValidationPredictions",
    doc = """Whether to keep the predictions of the cross-validation predictions. This needs to be set to TRUE if running the same AutoML object for repeated runs because CV predictions are required to build additional Stacked Ensemble models in AutoML.""")

  protected val keepCrossValidationModels = booleanParam(
    name = "keepCrossValidationModels",
    doc = """Whether to keep the cross-validated models. Keeping cross-validation models may consume significantly more memory in the H2O cluster.""")

  protected val keepCrossValidationFoldAssignment = booleanParam(
    name = "keepCrossValidationFoldAssignment",
    doc = """Whether to keep cross-validation assignments.""")

  protected val exportCheckpointsDir = nullableStringParam(
    name = "exportCheckpointsDir",
    doc = """Path to a directory where every generated model will be stored.""")

  protected val distribution = stringParam(
    name = "distribution",
    doc = """Distribution function used by algorithms that support it; other algorithms use their defaults. Possible values are ``"AUTO"``, ``"bernoulli"``, ``"quasibinomial"``, ``"modified_huber"``, ``"multinomial"``, ``"ordinal"``, ``"gaussian"``, ``"poisson"``, ``"gamma"``, ``"tweedie"``, ``"huber"``, ``"laplace"``, ``"quantile"``, ``"fractionalbinomial"``, ``"negativebinomial"``, ``"custom"``.""")

  protected val tweediePower = doubleParam(
    name = "tweediePower",
    doc = """Tweedie power for Tweedie regression, must be between 1 and 2.""")

  protected val quantileAlpha = doubleParam(
    name = "quantileAlpha",
    doc = """Desired quantile for Quantile regression, must be between 0 and 1.""")

  protected val huberAlpha = doubleParam(
    name = "huberAlpha",
    doc = """Desired quantile for Huber/M-regression (threshold between quadratic and linear loss, must be between 0 and 1).""")

  protected val customMetricFunc = nullableStringParam(
    name = "customMetricFunc",
    doc = """Reference to custom evaluation function, format: `language:keyName=funcName`.""")

  protected val customDistributionFunc = nullableStringParam(
    name = "customDistributionFunc",
    doc = """Reference to custom distribution, format: `language:keyName=funcName`.""")

  //
  // Default values
  //
  setDefault(
    projectName -> null,
    nfolds -> -1,
    balanceClasses -> false,
    classSamplingFactors -> null,
    maxAfterBalanceSize -> 5.0f,
    keepCrossValidationPredictions -> false,
    keepCrossValidationModels -> false,
    keepCrossValidationFoldAssignment -> false,
    exportCheckpointsDir -> null,
    distribution -> DistributionFamily.AUTO.name(),
    tweediePower -> 1.5,
    quantileAlpha -> 0.5,
    huberAlpha -> 0.9,
    customMetricFunc -> null,
    customDistributionFunc -> null)

  //
  // Getters
  //
  def getProjectName(): String = $(projectName)

  def getNfolds(): Int = $(nfolds)

  def getBalanceClasses(): Boolean = $(balanceClasses)

  def getClassSamplingFactors(): Array[Float] = $(classSamplingFactors)

  def getMaxAfterBalanceSize(): Float = $(maxAfterBalanceSize)

  def getKeepCrossValidationPredictions(): Boolean = $(keepCrossValidationPredictions)

  def getKeepCrossValidationModels(): Boolean = $(keepCrossValidationModels)

  def getKeepCrossValidationFoldAssignment(): Boolean = $(keepCrossValidationFoldAssignment)

  def getExportCheckpointsDir(): String = $(exportCheckpointsDir)

  def getDistribution(): String = $(distribution)

  def getTweediePower(): Double = $(tweediePower)

  def getQuantileAlpha(): Double = $(quantileAlpha)

  def getHuberAlpha(): Double = $(huberAlpha)

  def getCustomMetricFunc(): String = $(customMetricFunc)

  def getCustomDistributionFunc(): String = $(customDistributionFunc)

  //
  // Setters
  //
  def setProjectName(value: String): this.type = {
    set(projectName, value)
  }
           
  def setNfolds(value: Int): this.type = {
    set(nfolds, value)
  }
           
  def setBalanceClasses(value: Boolean): this.type = {
    set(balanceClasses, value)
  }
           
  def setClassSamplingFactors(value: Array[Float]): this.type = {
    set(classSamplingFactors, value)
  }
           
  def setMaxAfterBalanceSize(value: Float): this.type = {
    set(maxAfterBalanceSize, value)
  }
           
  def setKeepCrossValidationPredictions(value: Boolean): this.type = {
    set(keepCrossValidationPredictions, value)
  }
           
  def setKeepCrossValidationModels(value: Boolean): this.type = {
    set(keepCrossValidationModels, value)
  }
           
  def setKeepCrossValidationFoldAssignment(value: Boolean): this.type = {
    set(keepCrossValidationFoldAssignment, value)
  }
           
  def setExportCheckpointsDir(value: String): this.type = {
    set(exportCheckpointsDir, value)
  }
           
  def setDistribution(value: String): this.type = {
    val validated = EnumParamValidator.getValidatedEnumValue[DistributionFamily](value)
    set(distribution, validated)
  }
           
  def setTweediePower(value: Double): this.type = {
    set(tweediePower, value)
  }
           
  def setQuantileAlpha(value: Double): this.type = {
    set(quantileAlpha, value)
  }
           
  def setHuberAlpha(value: Double): this.type = {
    set(huberAlpha, value)
  }
           
  def setCustomMetricFunc(value: String): this.type = {
    set(customMetricFunc, value)
  }
           
  def setCustomDistributionFunc(value: String): this.type = {
    set(customDistributionFunc, value)
  }
           

  override private[sparkling] def getH2OAlgorithmParams(trainingFrame: H2OFrame): Map[String, Any] = {
    super.getH2OAlgorithmParams(trainingFrame) ++ getH2OAutoMLBuildControlParams(trainingFrame)
  }

  private[sparkling] def getH2OAutoMLBuildControlParams(trainingFrame: H2OFrame): Map[String, Any] = {
      Map(
        "project_name" -> getProjectName(),
        "nfolds" -> getNfolds(),
        "balance_classes" -> getBalanceClasses(),
        "class_sampling_factors" -> getClassSamplingFactors(),
        "max_after_balance_size" -> getMaxAfterBalanceSize(),
        "keep_cross_validation_predictions" -> getKeepCrossValidationPredictions(),
        "keep_cross_validation_models" -> getKeepCrossValidationModels(),
        "keep_cross_validation_fold_assignment" -> getKeepCrossValidationFoldAssignment(),
        "export_checkpoints_dir" -> getExportCheckpointsDir(),
        "distribution" -> getDistribution(),
        "tweedie_power" -> getTweediePower(),
        "quantile_alpha" -> getQuantileAlpha(),
        "huber_alpha" -> getHuberAlpha(),
        "custom_metric_func" -> getCustomMetricFunc(),
        "custom_distribution_func" -> getCustomDistributionFunc())
  }

  override private[sparkling] def getSWtoH2OParamNameMap(): Map[String, String] = {
    super.getSWtoH2OParamNameMap() ++
      Map(
        "projectName" -> "project_name",
        "nfolds" -> "nfolds",
        "balanceClasses" -> "balance_classes",
        "classSamplingFactors" -> "class_sampling_factors",
        "maxAfterBalanceSize" -> "max_after_balance_size",
        "keepCrossValidationPredictions" -> "keep_cross_validation_predictions",
        "keepCrossValidationModels" -> "keep_cross_validation_models",
        "keepCrossValidationFoldAssignment" -> "keep_cross_validation_fold_assignment",
        "exportCheckpointsDir" -> "export_checkpoints_dir",
        "distribution" -> "distribution",
        "tweediePower" -> "tweedie_power",
        "quantileAlpha" -> "quantile_alpha",
        "huberAlpha" -> "huber_alpha",
        "customMetricFunc" -> "custom_metric_func",
        "customDistributionFunc" -> "custom_distribution_func")
  }
      
}
