/*
 * Decompiled with CFR 0.152.
 */
package water.parser.parquet;

import java.time.Instant;
import org.apache.parquet.column.Dictionary;
import org.apache.parquet.io.api.Binary;
import org.apache.parquet.io.api.Converter;
import org.apache.parquet.io.api.GroupConverter;
import org.apache.parquet.io.api.PrimitiveConverter;
import org.apache.parquet.schema.DecimalMetadata;
import org.apache.parquet.schema.MessageType;
import org.apache.parquet.schema.OriginalType;
import org.apache.parquet.schema.PrimitiveType;
import org.apache.parquet.schema.Type;
import water.parser.BufferedString;
import water.parser.parquet.ParquetInt96TimestampConverter;
import water.parser.parquet.WriterDelegate;
import water.parser.parquet.ext.DecimalUtils;
import water.util.StringUtils;

class ChunkConverter
extends GroupConverter {
    private final WriterDelegate _writer;
    private final Converter[] _converters;
    private long _currentRecordIdx = -1L;

    ChunkConverter(MessageType parquetSchema, byte[] chunkSchema, WriterDelegate writer, boolean[] keepColumns) {
        this._writer = writer;
        int colIdx = 0;
        this._converters = new Converter[chunkSchema.length];
        int trueColumnIndex = 0;
        for (Type parquetField : parquetSchema.getFields()) {
            assert (parquetField.isPrimitive());
            if (keepColumns[trueColumnIndex]) {
                this._converters[trueColumnIndex] = this.newConverter(colIdx, chunkSchema[trueColumnIndex], parquetField.asPrimitiveType());
                ++colIdx;
            } else {
                this._converters[trueColumnIndex] = this.nullConverter(chunkSchema[trueColumnIndex], parquetField.asPrimitiveType());
            }
            ++trueColumnIndex;
        }
    }

    public Converter getConverter(int fieldIndex) {
        return this._converters[fieldIndex];
    }

    public void start() {
        ++this._currentRecordIdx;
        this._writer.startLine();
    }

    public void end() {
        this._writer.endLine();
    }

    long getCurrentRecordIdx() {
        return this._currentRecordIdx;
    }

    private PrimitiveConverter nullConverter(byte vecType, PrimitiveType parquetType) {
        switch (vecType) {
            case 0: 
            case 1: 
            case 2: 
            case 3: 
            case 4: 
            case 5: {
                boolean dictSupport = parquetType.getOriginalType() == OriginalType.UTF8 || parquetType.getOriginalType() == OriginalType.ENUM;
                return new NullStringConverter(dictSupport);
            }
        }
        throw new UnsupportedOperationException("Unsupported type " + vecType);
    }

    private PrimitiveConverter newConverter(int colIdx, byte vecType, PrimitiveType parquetType) {
        switch (vecType) {
            case 0: 
            case 2: 
            case 4: {
                if (parquetType.getPrimitiveTypeName().equals((Object)PrimitiveType.PrimitiveTypeName.BOOLEAN)) {
                    return new BooleanConverter(this._writer, colIdx);
                }
            }
            case 1: 
            case 5: {
                if (OriginalType.TIMESTAMP_MILLIS.equals((Object)parquetType.getOriginalType()) || parquetType.getPrimitiveTypeName().equals((Object)PrimitiveType.PrimitiveTypeName.INT96)) {
                    return new TimestampConverter(colIdx, this._writer);
                }
                if (OriginalType.DATE.equals((Object)parquetType.getOriginalType()) || parquetType.getPrimitiveTypeName().equals((Object)PrimitiveType.PrimitiveTypeName.INT32)) {
                    return new DateConverter(colIdx, this._writer);
                }
                boolean dictSupport = parquetType.getOriginalType() == OriginalType.UTF8 || parquetType.getOriginalType() == OriginalType.ENUM;
                return new StringConverter(this._writer, colIdx, dictSupport);
            }
            case 3: {
                if (OriginalType.DECIMAL.equals((Object)parquetType.getOriginalType())) {
                    return new DecimalConverter(colIdx, parquetType.getDecimalMetadata(), this._writer);
                }
                return new NumberConverter(colIdx, this._writer);
            }
        }
        throw new UnsupportedOperationException("Unsupported type " + vecType);
    }

    private static class DateConverter
    extends PrimitiveConverter {
        private static final long EPOCH_MILLIS = Instant.EPOCH.toEpochMilli();
        private static final long MILLIS_IN_A_DAY = 86400000L;
        private final int _colIdx;
        private final WriterDelegate _writer;

        DateConverter(int _colIdx, WriterDelegate _writer) {
            this._colIdx = _colIdx;
            this._writer = _writer;
        }

        public void addInt(int numberOfDaysFromUnixEpoch) {
            long parquetDateEpochMillis = EPOCH_MILLIS + (long)numberOfDaysFromUnixEpoch * 86400000L;
            this._writer.addNumCol(this._colIdx, parquetDateEpochMillis);
        }
    }

    private static class TimestampConverter
    extends PrimitiveConverter {
        private final int _colIdx;
        private final WriterDelegate _writer;

        TimestampConverter(int _colIdx, WriterDelegate _writer) {
            this._colIdx = _colIdx;
            this._writer = _writer;
        }

        public void addLong(long value) {
            this._writer.addNumCol(this._colIdx, value, 0);
        }

        public void addBinary(Binary value) {
            long timestampMillis = ParquetInt96TimestampConverter.getTimestampMillis(value);
            this._writer.addNumCol(this._colIdx, timestampMillis);
        }
    }

    private static class DecimalConverter
    extends PrimitiveConverter {
        private final int _colIdx;
        private final WriterDelegate _writer;
        private final int _precision;
        private final int _scale;

        DecimalConverter(int colIdx, DecimalMetadata dm, WriterDelegate writer) {
            this._colIdx = colIdx;
            this._precision = dm.getPrecision();
            this._scale = dm.getScale();
            this._writer = writer;
        }

        public void addBoolean(boolean value) {
            throw new UnsupportedOperationException("Boolean type is not supported by DecimalConverter");
        }

        public void addDouble(double value) {
            throw new UnsupportedOperationException("Double type is not supported by DecimalConverter");
        }

        public void addFloat(float value) {
            throw new UnsupportedOperationException("Float type is not supported by DecimalConverter");
        }

        public void addInt(int value) {
            this._writer.addNumCol(this._colIdx, value, -this._scale);
        }

        public void addLong(long value) {
            this._writer.addNumCol(this._colIdx, value, -this._scale);
        }

        public void addBinary(Binary value) {
            this._writer.addNumCol(this._colIdx, DecimalUtils.binaryToDecimal(value, this._precision, this._scale).doubleValue());
        }
    }

    private static class NumberConverter
    extends PrimitiveConverter {
        private final int _colIdx;
        private final WriterDelegate _writer;
        private final BufferedString _bs = new BufferedString();

        NumberConverter(int _colIdx, WriterDelegate _writer) {
            this._colIdx = _colIdx;
            this._writer = _writer;
        }

        public void addBoolean(boolean value) {
            this._writer.addNumCol(this._colIdx, value ? 1.0 : 0.0);
        }

        public void addDouble(double value) {
            this._writer.addNumCol(this._colIdx, value);
        }

        public void addFloat(float value) {
            this._writer.addNumCol(this._colIdx, value);
        }

        public void addInt(int value) {
            this._writer.addNumCol(this._colIdx, value, 0);
        }

        public void addLong(long value) {
            this._writer.addNumCol(this._colIdx, value, 0);
        }

        public void addBinary(Binary value) {
            this._bs.set(StringUtils.bytesOf(value.toStringUsingUTF8()));
            this._writer.addStrCol(this._colIdx, this._bs);
        }
    }

    private static class StringConverter
    extends PrimitiveConverter {
        private final BufferedString _bs = new BufferedString();
        private final int _colIdx;
        private final WriterDelegate _writer;
        private final boolean _dictionarySupport;
        private String[] _dict;

        StringConverter(WriterDelegate writer, int colIdx, boolean dictionarySupport) {
            this._colIdx = colIdx;
            this._writer = writer;
            this._dictionarySupport = dictionarySupport;
        }

        public void addBinary(Binary value) {
            this.writeStrCol(StringUtils.bytesOf(value.toStringUsingUTF8()));
        }

        public boolean hasDictionarySupport() {
            return this._dictionarySupport;
        }

        public void setDictionary(Dictionary dictionary) {
            this._dict = new String[dictionary.getMaxId() + 1];
            for (int i2 = 0; i2 <= dictionary.getMaxId(); ++i2) {
                this._dict[i2] = dictionary.decodeToBinary(i2).toStringUsingUTF8();
            }
        }

        public void addValueFromDictionary(int dictionaryId) {
            this.writeStrCol(StringUtils.bytesOf(this._dict[dictionaryId]));
        }

        private void writeStrCol(byte[] data) {
            this._bs.set(data);
            this._writer.addStrCol(this._colIdx, this._bs);
        }
    }

    private static class BooleanConverter
    extends PrimitiveConverter {
        private BufferedString TRUE = new BufferedString("True");
        private BufferedString FALSE = new BufferedString("False");
        private final int _colIdx;
        private final WriterDelegate _writer;

        BooleanConverter(WriterDelegate writer, int colIdx) {
            this._colIdx = colIdx;
            this._writer = writer;
        }

        public void addBoolean(boolean value) {
            BufferedString bsValue = value ? this.TRUE : this.FALSE;
            this._writer.addStrCol(this._colIdx, bsValue);
        }
    }

    private static class NullStringConverter
    extends PrimitiveConverter {
        private final boolean _dictionarySupport;

        NullStringConverter(boolean dictionarySupport) {
            this._dictionarySupport = dictionarySupport;
        }

        public void addBinary(Binary value) {
        }

        public boolean hasDictionarySupport() {
            return this._dictionarySupport;
        }

        public void setDictionary(Dictionary dictionary) {
        }

        public void addValueFromDictionary(int dictionaryId) {
        }

        public void addBoolean(boolean value) {
        }

        public void addDouble(double value) {
        }

        public void addFloat(float value) {
        }

        public void addInt(int value) {
        }

        public void addLong(long value) {
        }
    }
}

