package ai.koog.agents.core.feature.model.events

import ai.koog.agents.utils.ModelInfo
import ai.koog.prompt.dsl.ModerationResult
import ai.koog.prompt.dsl.Prompt
import ai.koog.prompt.message.Message
import kotlinx.datetime.Clock
import kotlinx.serialization.Serializable

/**
 * Represents an event indicating the start of a call to a Language Learning Model (LLM).
 *
 * This event captures the details of the LLM interaction at the point of invocation, including the
 * input prompt and any tools that will be used during the call. It extends the `DefinedFeatureEvent` class
 * and serves as a specific type of event in a feature-driven framework.
 *
 * @property runId A unique identifier associated with the specific run of the LLM call.
 * @property prompt The input prompt encapsulated as a [Prompt] object. This represents the structured set of
 *                  messages and configuration parameters sent to the LLM.
 * @property model The model information including provider, model, and context details.
 * @property tools The list of tools, represented by their string identifiers, being used within the scope
 *                 of the LLM call. These tools may extend or enhance the core functionality of the LLM.
 * @property timestamp The timestamp of the event, in milliseconds since the Unix epoch.
 */
@Serializable
public data class LLMCallStartingEvent(
    val runId: String,
    val prompt: Prompt,
    val model: ModelInfo,
    val tools: List<String>,
    override val timestamp: Long = Clock.System.now().toEpochMilliseconds(),
) : DefinedFeatureEvent() {

    /**
     * @deprecated Use constructor with model parameter of type [ModelInfo]:
     *             LLMCallStartingEvent(runId, prompt, model, tools, timestamp)
     */
    @Deprecated(
        message = "Please use constructor with model parameter of type [ModelInfo]: LLMCallStartingEvent(runId, prompt, model, tools, timestamp)",
        replaceWith = ReplaceWith("LLMCallStartingEvent(runId, prompt, model, tools, timestamp)")
    )
    public constructor(
        runId: String,
        prompt: Prompt,
        model: String,
        tools: List<String>,
        eventId: String = LLMCallStartingEvent::class.simpleName!!,
        timestamp: Long = Clock.System.now().toEpochMilliseconds()
    ) : this(runId, prompt, ModelInfo.fromString(model), tools, timestamp)

    /**
     * @deprecated Use model.eventString instead
     */
    @Deprecated("Use model.eventString instead", ReplaceWith("model.eventString"))
    public val modelString: String get() = model.eventString
}

/**
 * Represents an event signaling the completion of an LLM (Large Language Model) call.
 *
 * This event encapsulates the responses provided by the LLM during its operation. It serves as a
 * record of the responses generated by the LLM, marking the end of a particular interaction cycle.
 * The event is used within the system to capture relevant output data and ensure proper tracking
 * and logging of LLM-related interactions.
 *
 * @property runId The unique identifier of the LLM run.
 * @property prompt The input prompt encapsulated as a [Prompt] object. This represents the structured set of
 *                  messages and configuration parameters sent to the LLM.
 * @property model The model information including provider, model, and context details.
 * @property responses A list of responses generated by the LLM, represented as instances of [Message.Response].
 *                     Each response contains content, metadata, and additional context about the interaction.
 * @property moderationResponse The moderation response, if any, returned by the LLM.
 *                              This is typically used to capture and track content moderation results.
 * @property timestamp The timestamp of the event, in milliseconds since the Unix epoch.
 */
@Serializable
public data class LLMCallCompletedEvent(
    val runId: String,
    val prompt: Prompt,
    val model: ModelInfo,
    val responses: List<Message.Response>,
    val moderationResponse: ModerationResult? = null,
    override val timestamp: Long = Clock.System.now().toEpochMilliseconds(),
) : DefinedFeatureEvent() {

    /**
     * @deprecated Use constructor with model parameter of type [ModelInfo]:
     *             LLMCallCompletedEvent(runId, prompt, model, responses, moderationResponse, timestamp)
     */
    @Deprecated(
        message = "Please use constructor with model parameter of type [ModelInfo]: LLMCallCompletedEvent(runId, prompt, model, responses, moderationResponse, timestamp)",
        replaceWith = ReplaceWith("LLMCallCompletedEvent(runId, prompt, model, responses, moderationResponse, timestamp)")
    )
    public constructor(
        runId: String,
        prompt: Prompt,
        model: String,
        responses: List<Message.Response>,
        moderationResponse: ModerationResult? = null,
        eventId: String = LLMCallCompletedEvent::class.simpleName!!,
        timestamp: Long = Clock.System.now().toEpochMilliseconds()
    ) : this(runId, prompt, ModelInfo.fromString(model), responses, moderationResponse, timestamp)

    /**
     * @deprecated Use model.eventString instead
     */
    @Deprecated("Use model.eventString instead", ReplaceWith("model.eventString"))
    public val modelString: String get() = model.eventString
}

//region Deprecated

@Deprecated(
    message = "Use LLMCallStartingEvent instead",
    replaceWith = ReplaceWith("LLMCallStartingEvent")
)
public typealias BeforeLLMCallEvent = LLMCallStartingEvent

@Deprecated(
    message = "Use LLMCallCompletedEvent instead",
    replaceWith = ReplaceWith("LLMCallCompletedEvent")
)
public typealias AfterLLMCallEvent = LLMCallCompletedEvent

//endregion Deprecated
