package ai.koog.prompt.llm

import kotlinx.serialization.Serializable

/**
 * Represents a Large Language Model (LLM) with a specific provider, identifier, and a set of capabilities.
 *
 * @property provider The provider of the LLM, such as Google, OpenAI, or Meta.
 * @property id A unique identifier for the LLM instance. This typically represents the specific model version or name.
 * @property capabilities A list of capabilities supported by the LLM, such as temperature adjustment, tools usage, or schema-based tasks.
 * @property contextLength The context length of the LLM. This is the maximum number of tokens the LLM can process.
 * @property maxOutputTokens The maximum number of tokens that can be generated by the provider for the LLM.
 */
@Serializable
public data class LLModel(
    val provider: LLMProvider,
    val id: String,
    val capabilities: List<LLMCapability>,
    val contextLength: Long,
    val maxOutputTokens: Long? = null,
) {
    /**
     * Checks if the Large Language Model (LLM) supports a specific capability.
     *
     * @param capability The capability to check for, represented by an instance of [LLMCapability].
     * @return `true` if the LLM supports the given capability, otherwise `false`.
     */
    public fun supports(capability: LLMCapability): Boolean = capabilities.contains(capability)
}
