package ai.libs.jaicore.ml.dyadranking.algorithm;

import org.nd4j.linalg.api.ndarray.INDArray;
import org.nd4j.linalg.factory.Nd4j;
import org.nd4j.linalg.indexing.NDArrayIndex;
import org.nd4j.linalg.ops.transforms.Transforms;

/**
 * Implements the negative log likelihood (NLL) loss function for PL networks as described in [1].
 *  * 
 * [1]: Dirk Schäfer, Eyke Hüllermeier (2018). Dyad ranking using Plackett-Luce models based on joint feature representations
 * 
 * @author michael
 *
 */

public class PLNetLoss {
	
	private PLNetLoss() {
		//Intentionally left blank
	}
	
	/**
	 * Computes the NLL for PL networks according to equation (27) in [1].
	 * 
	 * @param 	plNetOutputs The outputs for M_n dyads generated by a PLNet's output layer in order of their ranking (from best to worst).
	 * @return	The NLL loss for the given PLNet outputs.
	 */
	public static INDArray computeLoss(INDArray plNetOutputs) {
		if (!(plNetOutputs.isRowVector()) || plNetOutputs.size(1) < 2 ) {
			throw new IllegalArgumentException("Input has to be a row vector of 2 or more elements.");
		}
		long dyadRankingLength = plNetOutputs.size(1);
		double loss = 0;
		for (int m = 0; m <= dyadRankingLength - 2; m++) {
			INDArray innerSumSlice = plNetOutputs.get(NDArrayIndex.interval(m, dyadRankingLength));
			innerSumSlice = Transforms.exp(innerSumSlice);
			loss += Transforms.log(innerSumSlice.sum(1)).getDouble(0);
		}
		loss -= plNetOutputs.get(NDArrayIndex.interval(0, dyadRankingLength - 1)).sum(1).getDouble(0);
		return Nd4j.create(new double[]{loss});
	}
	
	/**
	 * Computes the gradient of the NLL for PL networks w.r.t. the k-th dyad according to equation (28) in [1].
	 * @param plNetOutputs  The outputs for M_n dyads generated by a PLNet's output layer in order of their ranking (from best to worst).
	 * @param k				The ranking position with respect to which the gradient should be computed. Assumes zero-based indices, unlike the paper.
	 * @return				The gradient of the NLL loss w.r.t. the k-th dyad in the ranking.
	 */
	public static INDArray computeLossGradient(INDArray plNetOutputs, int k) {
		if (!(plNetOutputs.isRowVector()) || plNetOutputs.size(1) < 2 || k < 0 || k >= plNetOutputs.size(1)) {
			throw new IllegalArgumentException("Input has to be a row vector of 2 or more elements. And k has to be a valid index of plNetOutputs.");
		}
		long dyadRankingLength = plNetOutputs.size(1);
		double errorGradient = 0;
		for (int m = 0; m <= k; m++) {
			INDArray innerSumSlice = plNetOutputs.get(NDArrayIndex.interval(m, dyadRankingLength));
			innerSumSlice = Transforms.exp(innerSumSlice);
			double innerSum = innerSumSlice.sum(1).getDouble(0);
			errorGradient += Math.exp(plNetOutputs.getDouble(k)) / innerSum;
		}
		errorGradient -= 1;
		return Nd4j.create(new double[] {errorGradient});
	}

}
