package ai.libs.jaicore.search.algorithms.standard.uncertainty.explorationexploitationsearch;

import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Queue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ai.libs.jaicore.search.algorithms.standard.bestfirst.ENodeAnnotation;
import ai.libs.jaicore.search.algorithms.standard.uncertainty.ISolutionDistanceMetric;
import ai.libs.jaicore.search.model.travesaltree.BackPointerPath;
import ai.libs.jaicore.search.model.travesaltree.DefaultNodeComparator;

public class UncertaintyExplorationOpenSelection<T, A, V extends Comparable<V>> implements Queue<BackPointerPath<T, A, V>> {

	private static final Logger logger = LoggerFactory.getLogger(UncertaintyExplorationOpenSelection.class);

	private final Queue<BackPointerPath<T, A, V>> exploitationOpen = new PriorityQueue<>(new DefaultNodeComparator<>());
	private final PriorityQueue<BackPointerPath<T, A, V>> explorationOpen = new PriorityQueue<>(new DefaultNodeComparator<>());
	private final ISolutionDistanceMetric<T> solutionDistanceMetric;
	private final IPhaseLengthAdjuster phaseLengthAdjuster;
	private final IExplorationCandidateSelector<T,A, V> candidateSelector;
	private int explorationPhaseLength;
	private int exploitationPhaseLength;
	private double exploitationScoreThreshold;
	private double explorationUncertaintyThreshold;
	private int selectedNodes = 0;
	private int exploredNodes = 0;
	private long timeout;
	private long startTime;
	private boolean exploring = true;

	public UncertaintyExplorationOpenSelection(final long timeout, final int phaseInterval, final double exploitationScoreThreshold, final double explorationUncertaintyThreshold, final IPhaseLengthAdjuster phaseLengthAdjuster,
			final ISolutionDistanceMetric<T> solutionDistanceMetric, final IExplorationCandidateSelector<T, A,V> candidateSelector) {
		super();
		this.timeout = timeout;
		this.startTime = System.currentTimeMillis();
		int[] phaseLenghts = phaseLengthAdjuster.getInitialPhaseLengths(phaseInterval);
		assert phaseLenghts.length == 2;
		this.explorationPhaseLength = phaseLenghts[0];
		this.exploitationPhaseLength = phaseLenghts[1];
		this.exploitationScoreThreshold = exploitationScoreThreshold;
		this.explorationUncertaintyThreshold = explorationUncertaintyThreshold;
		this.phaseLengthAdjuster = phaseLengthAdjuster;
		this.solutionDistanceMetric = solutionDistanceMetric;
		this.candidateSelector = candidateSelector;
	}

	@Override
	public BackPointerPath<T, A, V> peek() {
		return this.selectCandidate(this.exploring);
	}

	private synchronized BackPointerPath<T, A, V> selectCandidate(final boolean isExploring) {
		Comparator<BackPointerPath<T, A, V>> comparator = (n1, n2) -> {
			try {
				Double u1 = (Double) n1.getAnnotation(ENodeAnnotation.F_UNCERTAINTY.name());
				Double u2 = (Double) n2.getAnnotation(ENodeAnnotation.F_UNCERTAINTY.name());
				V v1 = n1.getScore();
				V v2 = n2.getScore();
				if (isExploring) {
					if (Math.abs(u1 - u2) <= this.explorationUncertaintyThreshold) {
						return -1 * v1.compareTo(v2);
					} else {
						return Double.compare(u1, u2);
					}
				} else {
					if (v1 instanceof Double && v2 instanceof Double) {
						Double s1 = (Double) v1;
						Double s2 = (Double) v2;
						if (Math.abs(s1 - s2) <= this.exploitationScoreThreshold) {
							return Double.compare(u1, u2);
						} else {
							return Double.compare(s1, s2);
						}
					} else {
						if (v1 instanceof Double && v2 instanceof Double) {
							Double s1 = (Double) v1;
							Double s2 = (Double) v2;
							if (Math.abs(s1 - s2) <= this.exploitationScoreThreshold) {
								return Double.compare(u1, u2);
							} else {
								return v1.compareTo(v2);
							}
						} else {
							return v1.compareTo(v2);
						}
					}
				}
			} catch (Exception e) {
				logger.error(e.getMessage());
				return 0;
			}
		};
		if (isExploring) {
			return this.explorationOpen.stream().max(comparator).orElse(this.explorationOpen.peek());
		} else {
			return this.exploitationOpen.stream().min(comparator).orElse(this.exploitationOpen.peek());
		}
	}

	private void adjustPhaseLengths(final long passedTime) {
		int[] newPhaseLengths = this.phaseLengthAdjuster.adjustPhaseLength(this.explorationPhaseLength, this.exploitationPhaseLength, passedTime, this.timeout);
		assert newPhaseLengths.length == 2;
		this.explorationPhaseLength = newPhaseLengths[0];
		this.exploitationPhaseLength = newPhaseLengths[1];
	}

	@Override
	public synchronized boolean add(final BackPointerPath<T, A, V> node) {
		if (node == null) {
			throw new IllegalArgumentException("Cannot add node NULL to OPEN");
		}
		assert !this.contains(node) : "Node " + node + " is already there!";
		if (this.exploring) {
			return this.explorationOpen.add(node);
		} else {
			return this.exploitationOpen.add(node);
		}
	}

	@Override
	public synchronized boolean remove(final Object node) {
		if (this.exploitationOpen.contains(node) && this.explorationOpen.contains(node)) {
			throw new IllegalStateException("A node (" + node + ") that is to be removed is in BOTH open lists!");
		}
		if (this.exploring) {
			return this.explorationOpen.remove(node) || this.exploitationOpen.remove(node);
		} else {
			return this.exploitationOpen.remove(node) || this.explorationOpen.remove(node);
		}
	}

	@Override
	public synchronized boolean addAll(final Collection<? extends BackPointerPath<T, A, V>> arg0) {
		assert this.exploitationOpen != null : "Primary OPEN is NULL!";
		if (arg0 == null) {
			throw new IllegalArgumentException("Cannot add NULL collection");
		}
		if (this.exploring) {
			return this.explorationOpen.addAll(arg0);
		} else {
			return this.exploitationOpen.addAll(arg0);
		}
	}

	@Override
	public synchronized void clear() {
		this.exploitationOpen.clear();
		this.explorationOpen.clear();
	}

	@Override
	public synchronized boolean contains(final Object arg0) {
		return this.exploitationOpen.contains(arg0) || this.explorationOpen.contains(arg0);
	}

	@Override
	public boolean containsAll(final Collection<?> arg0) {
		for (Object o : arg0) {
			if (!this.contains(o)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public synchronized boolean isEmpty() {
		return this.exploitationOpen.isEmpty() && this.explorationOpen.isEmpty();
	}

	@Override
	public Iterator<BackPointerPath<T, A, V>> iterator() {
		return this.exploring ? this.explorationOpen.iterator() : this.exploitationOpen.iterator();
	}

	@Override
	public synchronized boolean removeAll(final Collection<?> arg0) {
		return this.exploitationOpen.removeAll(arg0) && this.explorationOpen.removeAll(arg0);
	}

	@Override
	public synchronized boolean retainAll(final Collection<?> arg0) {
		return this.exploitationOpen.retainAll(arg0) && this.explorationOpen.retainAll(arg0);
	}

	@Override
	public synchronized int size() {
		return this.exploitationOpen.size() + this.explorationOpen.size();
	}

	@Override
	public Object[] toArray() {
		return this.exploitationOpen.toArray();
	}

	@SuppressWarnings("unchecked")
	@Override
	public <X> X[] toArray(final X[] arg0) {
		return (X[]) this.exploitationOpen.toArray();
	}

	@Override
	public BackPointerPath<T, A, V> element() {
		return this.peek();
	}

	@Override
	public boolean offer(final BackPointerPath<T, A, V> e) {
		return this.add(e);
	}

	@Override
	public BackPointerPath<T, A, V> poll() {
		return this.remove();
	}

	@Override
	public synchronized BackPointerPath<T, A, V> remove() {

		/* determine element to be returned and remove it from the respective list */
		BackPointerPath<T, A, V> peek = this.peek();
		this.remove(peek);

		/* update internal state */
		if (!this.exploring) {
			this.selectedNodes++;
			if (this.selectedNodes % this.exploitationPhaseLength == 0) {
				List<BackPointerPath<T, A, V>> explorationCandidates = this.candidateSelector.selectExplorationCandidates(this.exploitationOpen, this.exploitationOpen.peek(), this.solutionDistanceMetric);

				/* enable exploration with the node selected by the explorer evaluator */
				try {
					logger.info("Entering exploration phase under {}", explorationCandidates);
				} catch (Exception e) {
					logger.error(e.getMessage());
				}
				this.exploring = true;
				this.exploredNodes = 0;
				this.exploitationOpen.removeAll(explorationCandidates);
				this.explorationOpen.clear();
				this.explorationOpen.addAll(explorationCandidates);
			}
		} else {
			this.exploredNodes++;
			if (this.exploredNodes > this.explorationPhaseLength || this.explorationOpen.isEmpty()) {
				this.adjustPhaseLengths(System.currentTimeMillis() - this.startTime);
				this.exploring = false;
				this.exploitationOpen.addAll(this.explorationOpen);
				this.explorationOpen.clear();
				logger.info("Entering exploitation phase");
			}
		}

		/* return the peeked element */
		return peek;
	}
}
