package ai.libs.jaicore.search.model.travesaltree;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.api4.java.datastructure.graph.implicit.IGraphGenerator;
import org.api4.java.datastructure.graph.implicit.INewNodeDescription;
import org.api4.java.datastructure.graph.implicit.IRootGenerator;
import org.api4.java.datastructure.graph.implicit.ISuccessorGenerator;

/**
 * Graph generator that uses another graph generator as a basis by reducing the
 * graph generated by the basis generator so that it does not contain long
 * chains of nodes anymore, that is successors of a node are skipped while a
 * node only has 1 successor. Usage: Create a new ReducedGraphGenerator object
 * and swap all ocurrences of the graph generator it uses as a basis with the
 * reduced graph generator.
 *
 * @author Helena Graf
 *
 * @param <T>
 * @param <A>
 */
public class ReducedGraphGenerator<T, A> implements IGraphGenerator<T, A> {

	private IGraphGenerator<T, A> basis;

	/**
	 * Create a new ReducedGraphGenerator that uses the given graph generator as a
	 * basis.
	 *
	 * @param basis
	 *            the graph generator to use as a basis
	 */
	public ReducedGraphGenerator(final IGraphGenerator<T, A> basis) {
		this.basis = basis;
	}

	@Override
	public IRootGenerator<T> getRootGenerator() {
		return this.basis.getRootGenerator();
	}

	@Override
	public ISuccessorGenerator<T, A> getSuccessorGenerator() {
		return new ISuccessorGenerator<T, A>() {

			private ISuccessorGenerator<T, A> generator = ReducedGraphGenerator.this.basis.getSuccessorGenerator();

			/**
			 * Expands the node recursively while it only has one successor, until the end
			 * of the branch or a split point in the graph is reached.
			 *
			 * @param node
			 *            The node to expand
			 * @return The fully refined node
			 * @throws InterruptedException
			 */
			public INewNodeDescription<T, A> reduce(final INewNodeDescription<T, A> node)
					throws InterruptedException {
				List<INewNodeDescription<T, A>> sucessors = this.generator.generateSuccessors(node.getTo());
				List<INewNodeDescription<T, A>> previous = Arrays.asList(node);
				while (sucessors.size() == 1) {
					previous = sucessors;
					sucessors = this.generator.generateSuccessors(sucessors.get(0).getTo());
				}
				return previous.get(0);
			}

			@Override
			public List<INewNodeDescription<T, A>> generateSuccessors(final T node) throws InterruptedException {
				List<INewNodeDescription<T, A>> successors = this.generator.generateSuccessors(node);
				// Skip through nodes with 1 successor to find
				while (successors.size() == 1) {
					List<INewNodeDescription<T, A>> previous = successors;
					successors = this.generator.generateSuccessors(successors.get(0).getTo());
					if (successors.isEmpty()) {
						return previous;
					}
				}

				List<INewNodeDescription<T, A>> reducedSuccessors = new ArrayList<>();
				for (INewNodeDescription<T, A> successor : successors) {
					reducedSuccessors.add(this.reduce(successor));
				}
				return reducedSuccessors;
			}
		};
	}
}
