package ai.lilystyle.analytics_android;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import org.jetbrains.annotations.NotNull;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;


/**
 * Core class for integrating Lily.Ai analytics.
 *
 * <p>First obtain api-key and token to access Lily.Ai analytics.
 * <p>Place the obtained api-key and token in meta-data tags in your AndroidManifest.xml inside application tag:
 * <pre>
 * {@code
 * <application ...>
 *     <meta-data android:name="ai.lily.analytics.api_token" android:value="PLACE_YOUR_API_TOKEN_HERE" />
 *     <meta-data android:name="ai.lily.analytics.api_key" android:value="PLACE_YOUR_API_KEY_HERE" />
 *     <meta-data android:name="ai.lily.analytics.endpoint" android:value="PLACE_YOUR_ENDPOINT_HERE" />
 * </application>
 * }
 * </pre>
 *
 * <p>Now you can call {@link #getInstance(Context)} to obtain LilyAi instance
 *
 * <p>You might want to keep your api-key and token more secure. In this case obfuscate those as you please and
 * to obtain LilyAi instance use {@link #getInstance(Context, String)} with your token or {@link #getInstance(Context, String, String, String)} with both.
 *
 * <p>When initiated LilyAi tries to restore a unique id (UUID or lpid) of the user from it's shared preferences or
 * generates a new one, if not present, and stores it. You can override this by calling {@link #setUUID(String)} with your own id for the user.
 * Also you can use {@link #getUUID()} to check what UUID is currently used by LilyAi or call {@link #resetUUID()} to
 * drop the stored UUID value and generate a new random UUID.
 *
 * <p>Once you obtained an instance of LilyAi and done with setting up the UUID you can call either of {@link #track(JSONObject)} or
 * {@link #track(JSONObject, LilyAiListener)} methods to send your data.
 */
public final class LilyAi {

    private final static Map<String, LilyAi> instancesMap = new HashMap<>();

    private final SharedPreferences prefs;
    private final String baseUrl;
    private final Map<String, String> headers = new ConcurrentHashMap<>();
    private final TrackingDataPersistentStorage trackingDataStorage;
    private long lastTrackId;
    private final String token;
    private final String apiKey;
    private final ExecutorService executorService;
    private String uuid;
    private String sessionId;
    private Long sessionStartTime;
    private Long sessionLastEventOccurred;
    private long sessionDuration;
    private final OkHttpClient okHttpClient = RequestsHelper.getOkHttpClient();
    private final Handler handler;


    private LilyAi(Context context, String baseUrl, String token, String apiKey) {
        this.baseUrl = baseUrl;
        this.token = token;
        this.apiKey = apiKey;
        prefs = context.getApplicationContext().getSharedPreferences(Constants.PREFS_FILE_NAME+(apiKey+token+baseUrl).hashCode(), Context.MODE_PRIVATE);
        executorService = new ThreadPoolExecutor(Constants.THREAD_POOL_SIZE, Constants.THREAD_POOL_SIZE,
                1, TimeUnit.MINUTES, new LinkedBlockingDeque<Runnable>(), new WorkerThreadFactory());
        uuid = Utils.getUUID(prefs, true);
        sessionId = prefs.getString(Constants.SESSION_ID_PREF_NAME, null);
        sessionStartTime = prefs.getLong(Constants.SESSION_START_PREF_NAME, 0);
        sessionLastEventOccurred = prefs.getLong(Constants.SESSION_LAST_EVENT_PREF_NAME, 0);
        sessionDuration = prefs.getLong(Constants.SESSION_DURATION, Constants.DEFAULT_SESSION_DURATION);
        lastTrackId = prefs.getLong(Constants.LAST_TRACKED_ID, 0);
        trackingDataStorage = TrackingDataPersistentStorage.getInstance(context.getApplicationContext(),
                String.format(Locale.getDefault(), Constants.TRACKED_DATA_STORAGE_DIR, baseUrl.hashCode()),
                new TrackingDataPersistentStorageListener() {
                    @Override
                    public void withUndeliveredData(TrackingData trackingData) {
                        synchronized (trackingDataStorage) {
                            if (!trackingDataStorage.isSending(trackingData)) {
                                trackingDataStorage.put(trackingData, true);
                                executorService.submit(new Worker(LilyAi.this.baseUrl, trackingData, true, okHttpClient, workerListener, null));
                            }
                        }
                    }
        });
        handler = new Handler(Looper.getMainLooper());
    }

    /**
     * Get instance of LilyAI.
     *
     * <p>To use this method you should provide api-key and token in meta-data tags in your AndroidManifest.xml inside application tag:
     * <pre>
     * {@code
     * <application ...>
     *     <meta-data android:name="ai.lily.analytics.api_token" android:value="PLACE_YOUR_API_TOKEN_HERE" />
     *     <meta-data android:name="ai.lily.analytics.api_key" android:value="PLACE_YOUR_API_KEY_HERE" />
     *     <meta-data android:name="ai.lily.analytics.endpoint" android:value="PLACE_YOUR_ENDPOINT_HERE" />
     * </application>
     * }
     * </pre>
     *
     * @param context The application context.
     * @return an instance of LilyAI associated with provided token and api-key.
     */
    public static LilyAi getInstance(Context context) {
        return getInstance(context, null, null, null);
    }

    /**
     * Get instance of LilyAI.
     *
     * <p>To use this method you should provide api-key in meta-data tag in your AndroidManifest.xml inside application tag:
     * <pre>
     * {@code
     * <application ...>
     *     <meta-data android:name="ai.lily.analytics.api_key" android:value="PLACE_YOUR_API_KEY_HERE" />
     *     <meta-data android:name="ai.lily.analytics.endpoint" android:value="PLACE_YOUR_ENDPOINT_HERE" />
     * </application>
     * }
     * </pre>
     *
     * @param context The application context.
     * @param token Your api token string.
     * @return an instance of LilyAI associated with provided token and api-key.
     */
    public static LilyAi getInstance(Context context, String token) {
        return getInstance(context, null, token, null);
    }

    /**
     * Get instance of LilyAI.
     *
     * @param context The application context.
     * @param token Your api token string.
     * @param apiKey Your api-key string.
     * @return an instance of LilyAI associated with provided token and api-key.
     */
    public static LilyAi getInstance(Context context, String baseUrl, String token, String apiKey) {
        if (context == null) {
            Log.e(Constants.LOG_TAG, "Context can't be null.");
            return null;
        }

        if (baseUrl == null) {
            baseUrl = Utils.getMetadataString(context, Constants.META_BASE_URL_KEY);
            if (baseUrl == null) {
                Log.e(Constants.LOG_TAG, "Application meta-data " + Constants.META_BASE_URL_KEY +
                        " is not set. Set it or provide it with getInstance() method.");
                return null;
            }
        }

        if (token == null) {
            token = Utils.getMetadataString(context, Constants.META_API_TOKEN_KEY);
            if (token == null) {
                Log.e(Constants.LOG_TAG, "Application meta-data " + Constants.META_API_TOKEN_KEY +
                        " is not set. Set it or provide it with getInstance() method.");
                return null;
            }
        }

        if (apiKey == null) {
            apiKey = Utils.getMetadataString(context, Constants.META_API_KEY_KEY);
            if (apiKey == null) {
                Log.e(Constants.LOG_TAG, "Application meta-data " + Constants.META_API_KEY_KEY +
                        " is not set. Set it or provide it with getInstance() method.");
                return null;
            }
        }

        synchronized (instancesMap) {
            LilyAi instance = instancesMap.get(apiKey+token+baseUrl);
            if (instance == null) {
                instance = new LilyAi(context, baseUrl, token, apiKey);
                instancesMap.put(apiKey+token+baseUrl, instance);
            }
            return instance;
        }
    }

    /**
     * Reset the stored UUID to a new random value.
     */
    public void resetUUID() {
        uuid = Utils.getUUID(prefs, false);
        sessionId = null;
        prefs.edit().putString(Constants.SESSION_ID_PREF_NAME, null).apply();
    }

    /**
     * Set a new custom UUID value.
     *
     * @param uuid New UUID value to store and use. Can't be null or empty string.
     */
    public void setUUID(@NotNull String uuid) {
        if (uuid.isEmpty()) {
            Log.e(Constants.LOG_TAG, "LilyAI UUID can't be null or empty string! Ignoring setUUID() call with argument "+ uuid);
            return;
        }
        if (!uuid.equals(this.uuid)) {
            sessionId = null;
            prefs.edit().putString(Constants.SESSION_ID_PREF_NAME, null).apply();
        }
        this.uuid = uuid;
        Utils.updateUUID(prefs, uuid);
    }

    /**
     * Get current UUID value stored by LilyAI.
     *
     * @return UUID string
     */
    public String getUUID() {
        return uuid;
    }

    /**
     * Set session duration time.
     *
     * @param sessionDuration max session duration time (millis)
     */
    public void setSessionDuration(long sessionDuration) {
        this.sessionDuration = sessionDuration;
        prefs.edit().putLong(Constants.SESSION_DURATION, sessionDuration).apply();
    }

    /**
     * Send your data to LilyAI.
     * Data is sent asynchronously.
     * Same as {@link #track(JSONObject, LilyAiListener)} with listener set to null.
     *
     * @param data Your JSON data.
     */
    public void track(JSONObject data) {
        track(data, null);
    }

    /**
     * Send your data to LilyAI.
     * Data is sent asynchronously.
     * Methods of {@link LilyAiListener} will be called on the MainLooper.
     *
     * @param data Your JSON data. If data is null {@link LilyAiListener#onError(String)} will be called with message "JSON data is null".
     * @param listener Callbacks for data send success or error. Can be null.
     */
    public void track(JSONObject data, final LilyAiListener listener) {
        if (data == null) {
            if (listener != null) {
                handler.post(new Runnable() {
                    @Override
                    public void run() {
                        listener.onError("JSON data is null");
                    }
                });
            }
            return;
        }
        if (sessionLastEventOccurred != null && sessionLastEventOccurred > 0 && System.currentTimeMillis() - sessionLastEventOccurred > sessionDuration) {
            sessionId = null;
        }
        if (sessionId == null) {
            sessionId = getUUID() + "-" + System.currentTimeMillis();
            sessionStartTime = System.currentTimeMillis();
            prefs.edit().putString(Constants.SESSION_ID_PREF_NAME, sessionId)
                    .putLong(Constants.SESSION_START_PREF_NAME, sessionStartTime)
                    .apply();
        }
        addHeader("lsid", sessionId);
        addHeader("lsstart", (sessionStartTime != null && sessionStartTime > 0) ? sessionStartTime.toString() : null);
        addHeader("lsend", String.valueOf(System.currentTimeMillis() + sessionDuration));
        sessionLastEventOccurred = System.currentTimeMillis();
        prefs.edit().putLong(Constants.SESSION_LAST_EVENT_PREF_NAME, sessionLastEventOccurred).apply();
        addHeader("x-api-key", apiKey);
        addHeader("Api-Token", token);
        addHeader("lpid", uuid);
        synchronized (trackingDataStorage) {
            TrackingData newData = new TrackingData(lastTrackId++, headers, data);
            prefs.edit().putLong(Constants.LAST_TRACKED_ID, lastTrackId).apply();
            trackingDataStorage.put(newData, true);
            executorService.submit(new Worker(baseUrl, newData, false, okHttpClient, workerListener, listener));
            trackingDataStorage.getNotDelivered(-1, Constants.THREAD_POOL_SIZE);
        }
    }

    public void addHeader(String name, String value) {
        if (name != null) {
            if (value == null) {
                headers.remove(name);
            } else {
                headers.put(name, value);
            }
        }
    }

    public void setUserID(String uid) {
        addHeader("uid", uid);
    }

    public void setAnalyticsProviderID(String aid) {
        addHeader("aid", aid);
    }

    public void setAnalyticsSessionID(String sid) {
        addHeader("sid", sid);
    }

    public void setHashedUserEmail(String uem) {
        addHeader("uem", uem);
    }

    public void setReferer(String src, String mdm, String pgpath, String pgcat, String pgtype) {
        try {
            JSONObject referer = new JSONObject();
            if (src != null) {
                referer.put("src", src);
            }
            if (mdm != null) {
                referer.put("mdm", mdm);
            }
            if (pgpath != null) {
                referer.put("pgpath", pgpath);
            }
            if (pgcat != null) {
                referer.put("pgcat", pgcat);
            }
            if (pgtype != null) {
                referer.put("pgtype", pgtype);
            }
            if (src != null || mdm != null || pgpath != null || pgcat != null || pgtype != null) {
                addHeader("referrer_info", referer.toString());
            } else {
                addHeader("referrer_info", null);
            }
        } catch (Exception e) {
            addHeader("referrer_info", null);
        }
    }

    public void setExperimentId(String expid) {
        addHeader("expid", expid);
    }

    public void setUserSourceIP(String sip) {
        addHeader("sip", sip);
    }

    public void setVisitorId(String vid) {
        addHeader("vid", vid);
    }

    private final WorkerListener workerListener = new WorkerListener() {
        @Override
        public void onSuccess(TrackingData data, boolean isFromRetry, final LilyAiListener lilyAiListener) {
            synchronized (trackingDataStorage) {
                trackingDataStorage.delivered(data);
                if (isFromRetry) {
                    trackingDataStorage.getNotDelivered(data.id, 1);
                }
            }
            if (lilyAiListener != null) {
                handler.post(new Runnable() {
                    @Override
                    public void run() {
                        lilyAiListener.onSuccess();
                    }
                });
            }
        }

        @Override
        public void onError(TrackingData data, boolean isFromRetry, int code, final String message, final LilyAiListener lilyAiListener) {
            synchronized (trackingDataStorage) {
                if (code <= 0 || (code >= 500 && code < 600)) {
                    trackingDataStorage.put(data, false);
                } else {
                    trackingDataStorage.delivered(data);
                    if (isFromRetry) {
                        trackingDataStorage.getNotDelivered(data.id, 1);
                    }
                }
            }

            if (lilyAiListener != null) {
                handler.post(new Runnable() {
                    @Override
                    public void run() {
                        lilyAiListener.onError(message);
                    }
                });
            }
        }
    };
}
