package ai.lilystyle.analytics_android;

import android.content.Context;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

class TrackingDataPersistentStorage extends Thread {
    private final Context context;
    private final Set<Long> sendingData = new HashSet<>();
    private final Set<TrackingData> trackingData = new HashSet<>();
    private final Set<Long> deliveredData = new HashSet<>();
    private final String filesDir;
    private volatile boolean notifyWithNotDeliveredData = true;
    private volatile long lastDataId = -1;
    private volatile int nextDataCount = 2;
    private final Set<TrackingDataPersistentStorageListener> listeners = new HashSet<>();
    private final static Map<String, TrackingDataPersistentStorage> instancesMap = new HashMap<>();

    private TrackingDataPersistentStorage(Context context, String filesDir) {
        this.context = context;
        this.filesDir = filesDir;
        setName(String.format(Locale.getDefault(), Constants.THREAD_NAME, 0));
        start();
    }

    public static TrackingDataPersistentStorage getInstance(Context context, String filesDir, TrackingDataPersistentStorageListener listener) {
        TrackingDataPersistentStorage instance;
        synchronized (instancesMap) {
            instance = instancesMap.get(filesDir);
            if (instance == null) {
                instance = new TrackingDataPersistentStorage(context, filesDir);
                instancesMap.put(filesDir, instance);
            }
        }
        instance.addListener(listener);
        return instance;
    }

    private void addListener(TrackingDataPersistentStorageListener listener) {
        synchronized (listeners) {
            listeners.add(listener);
        }
    }

    void put(TrackingData data, boolean isSending) {
        synchronized (trackingData) {
            trackingData.add(data);
            if (isSending) {
                sendingData.add(data.id);
            } else {
                sendingData.remove(data.id);
            }
            deliveredData.remove(data.id);
            trackingData.notifyAll();
        }
    }

    void delivered(TrackingData data) {
        synchronized (trackingData) {
            sendingData.remove(data.id);
            trackingData.remove(data);
            deliveredData.add(data.id);
            trackingData.notifyAll();
        }
    }

    void getNotDelivered(long lastId, int count) {
        synchronized (trackingData) {
            lastDataId = lastId;
            nextDataCount = count > 0 ? count : 1;
            notifyWithNotDeliveredData = true;
            trackingData.notifyAll();
        }
    }

    boolean isSending(TrackingData data) {
        synchronized (trackingData) {
            return sendingData.contains(data.id);
        }
    }

    @Override
    public void run() {
        while (true) {
            Set<TrackingData> dataToSave = new HashSet<>();
            Set<Long> dataToRemove = new HashSet<>();
            synchronized (trackingData) {
                for (Iterator<TrackingData> i = trackingData.iterator(); i.hasNext();) {
                    TrackingData data = i.next();
                    if (deliveredData.contains(data.id)) {
                        i.remove();
                    }
                }
                dataToSave.addAll(trackingData);
                dataToRemove.addAll(deliveredData);
                if (!notifyWithNotDeliveredData && dataToSave.isEmpty() && dataToRemove.isEmpty()) {
                    try {
                        trackingData.wait();
                        continue;
                    } catch (InterruptedException e) {
                        break;
                    }
                }
            }

            for (TrackingData data : dataToSave) {
                saveToDisk(data);
            }
            for (Long id : dataToRemove) {
                removeData(id);
            }
            if (!dataToSave.isEmpty() || dataToRemove.isEmpty()) {
                synchronized (trackingData) {
                    trackingData.removeAll(dataToSave);
                    deliveredData.removeAll(dataToRemove);
                }
            }

            if (notifyWithNotDeliveredData) {
                Set<Long> nowSending = new HashSet<>();
                synchronized (trackingData) {
                    notifyWithNotDeliveredData = false;
                    nowSending.addAll(sendingData);
                }

                Set<TrackingData> undeliveredData = restoreFromDisk(lastDataId, nextDataCount, nowSending);

                synchronized (listeners) {
                    for (TrackingData data : undeliveredData) {
                        for (TrackingDataPersistentStorageListener listener : listeners) {
                            listener.withUndeliveredData(data);
                        }
                    }
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    private Set<TrackingData> restoreFromDisk(long lastDataId, int nextDataCount, Set<Long> nowSending) {
        Set<TrackingData> undeliveredData = new HashSet<>();

        String fileNames[] = new File(context.getFilesDir(), filesDir).list();
        List<Long> ids = new ArrayList<>();
        if (fileNames != null) {
            for (String name : fileNames) {
                long id;
                try {
                    id = Long.parseLong(name);
                } catch (Exception e) {
                    continue;
                }
                ids.add(id);
            }
        }
        Collections.sort(ids);
        for (Long id : ids) {
            if (id > lastDataId && !nowSending.contains(id)) {
                TrackingData data = restoreFromDiskOrDelete(id);
                if (data != null) {
                    undeliveredData.add(data);
                    nextDataCount--;
                }
            }
            if (nextDataCount <= 0) {
                break;
            }
        }
        return undeliveredData;
    }

    private TrackingData restoreFromDiskOrDelete(long id) {
        File file = new File (new File(context.getFilesDir(), filesDir), String.valueOf(id));
        if (file.isFile()) {
            try (FileInputStream is = new FileInputStream(file); ObjectInputStream ois = new ObjectInputStream(is)) {
                return (TrackingData) ois.readObject();
            } catch (Exception e) {
                file.delete();
            }
        }
        return null;
    }

    private void saveToDisk(TrackingData data) {
        File file = new File (new File(context.getFilesDir(), filesDir), String.valueOf(data.id));
        if (!file.exists()) {
            File parent = file.getParentFile();
            if (parent != null) {
                file.getParentFile().mkdirs();
            }
            try (FileOutputStream os = new FileOutputStream(file); ObjectOutputStream oos = new ObjectOutputStream(os)) {
                oos.writeObject(data);
            } catch (Exception ignored) {}
        }
    }

    private void removeData(long id) {
        File file = new File (new File(context.getFilesDir(), filesDir), String.valueOf(id));
        file.delete();
    }
}
