package com.mobid.android

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import com.mobid.android.http.RemoteError
import com.mobid.android.http.isSuccess
import com.mobid.android.http.repository.MobIdRepository
import com.mobid.android.http.repository.MobIdSession
import com.mobid.android.model.ConferenceStatus
import com.mobid.android.utils.StatusUpdater
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch

internal class SessionManager private constructor() {

    private val job = Job()
    private val scope: CoroutineScope = CoroutineScope(job + Dispatchers.Main)

    private val repo by lazy { MobIdRepository.getInstance() }
    private var updater: StatusUpdater? = null

    val session = MutableLiveData<MobIdSession>()
    val error = MutableLiveData<RemoteError>()

    fun startNewSession(): LiveData<Boolean> = MutableLiveData<Boolean>().also { isLoading ->
        scope.launch(Dispatchers.IO) {
            isLoading.postValue(true)
            val result = repo.verification(ConferenceStatus.WAIT_INVITE)
            isLoading.postValue(false)
            scope.launch(Dispatchers.Main) {
                when {
                    result.isSuccess() -> {
                        session.postValue(result.data)
                        error.postValue(result.remoteError)
                        observeSession(result.data!!)
                    }
                    else -> error.postValue(result.remoteError)
                }
            }
        }
    }

    fun observeSession(session: MobIdSession){
        updater?.stop()
        updater = StatusUpdater(
            sessionCallBack = { this.session.postValue(it)},
            verificationId = session.verificationId!!,
            token = session.token!!,
            repo = repo,
            scope = scope
        ).begin()
    }

    fun stopObserving(){
        updater?.stop()
    }

    fun clear(){
        session.postValue(null)
        updater?.stop()
//        job.cancel()
    }


    companion object{

        private var instance: SessionManager? = null

        fun init(): SessionManager{
            if(instance == null){
                instance = SessionManager()
            }
            return instance!!
        }

    }
}