package com.mobid.android.http

import android.annotation.SuppressLint
import android.util.Log
import com.mobid.android.http.HttpMethod.DELETE
import com.mobid.android.http.HttpMethod.GET
import com.mobid.android.utils.getString
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import org.json.JSONTokener
import java.io.*
import java.net.*
import javax.net.ssl.HttpsURLConnection
import javax.net.ssl.SSLHandshakeException

internal class HttpClient {

    @SuppressLint("LogNotTimber")
    suspend fun call(
        verb: String,
        url: String,
        input: JSONObject? = null,
        token: String? = null
    ): RemoteResult<JSONObject> {

        var result = RemoteResult<JSONObject>()

        var urlConnection: HttpsURLConnection? = null
        var os: OutputStream? = null
        var writer: BufferedWriter? = null
        var ins: InputStream? = null
        try {
            val url1 = URL(URL + url)
            urlConnection = url1.openConnection() as HttpsURLConnection
            urlConnection.connectTimeout = 10000
            urlConnection.readTimeout = 10000
            urlConnection.doOutput = !(verb == GET || verb == DELETE)
            urlConnection.requestMethod = verb

            if (token != null) {
                urlConnection.setRequestProperty("Authorization", "Bearer $token")
            }
            urlConnection.setRequestProperty("Content-type", "application/json")
            urlConnection.setRequestProperty("Accept", "application/json")

            // disable caching
            urlConnection.defaultUseCaches = false
            urlConnection.useCaches = false

            if (input != null) {
                os = urlConnection.outputStream
                writer = BufferedWriter(OutputStreamWriter(os, "UTF-8"))
                writer.write(input.toString())
                writer.flush()
            }

            ins = try
            //temp fix for 401 and WWW-AUTH header bug
            {
                BufferedInputStream(urlConnection.inputStream)
            } catch (e: IOException) {
                BufferedInputStream(urlConnection.errorStream)
            }

            var outputJson = ins.getString()

            val httpResponseCode = urlConnection.responseCode

            val json = JSONTokener(outputJson).nextValue()
            if (json is JSONArray) {
                outputJson = "{\"value\":$outputJson}"
            }

            if (httpResponseCode in listOf(
                    HttpURLConnection.HTTP_CREATED,
                    HttpURLConnection.HTTP_OK,
                    HttpURLConnection.HTTP_ACCEPTED
                )
            ) {
                result.data = JSONObject(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_NOT_FOUND) {
                result.remoteError = RemoteError.HttpNotFound(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_GATEWAY_TIMEOUT) {
                result.remoteError = RemoteError.HttpTimeout(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_BAD_GATEWAY) {
                result.remoteError = RemoteError.HttpBadGetaway(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_FORBIDDEN) {
                result.remoteError = RemoteError.HttpForbidden(outputJson)
            } else if (httpResponseCode == HTTP_UNPROCESSABLE_ENTITY) {
                result.remoteError = RemoteError.HttpUnprocessableEntitiy(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_UNAUTHORIZED) {
                result.remoteError = RemoteError.HttpUnauthorized(outputJson)
            } else {
                result.remoteError = RemoteError.HttpUnauthorized(outputJson)
            }
        } catch (e: SSLHandshakeException) {
            Log.e(TAG, "Exception", e)
            result.remoteError = RemoteError.SslException(null)
        } catch (e: JSONException) {
            Log.e(TAG, "Exception", e)
            result.remoteError = RemoteError.JsonException(null)
        } catch (e: MalformedURLException) {
            Log.e(TAG, "Exception", e)
            result.remoteError = RemoteError.MalformedUrlException(null)
        } catch (e: UnsupportedEncodingException) {
            Log.e(TAG, "Exception", e)
            result.remoteError = RemoteError.UnsupportedEncodingException(null)
        } catch (e: ProtocolException) {
            Log.e(TAG, "Exception", e)
            result.remoteError = RemoteError.NetProtocolException(null)
        } catch (e: SocketTimeoutException) {
            Log.e(TAG, "Exception", e)
            result.remoteError = RemoteError.HttpTimeout(null)
        } catch (e: IOException) {
            Log.e(TAG, "Exception", e)
            result.remoteError = RemoteError.NoInternetException(null)
        } catch (e: Exception) {
            Log.e(TAG, "Exception", e)
            result.remoteError = RemoteError.Exception(null)
        } finally {
            try {
                writer?.close()
            } catch (e: IOException) {
                Log.e(TAG, "Exception", e)
            }

            try {
                ins?.close()
            } catch (e: IOException) {
                Log.e(TAG, "Exception", e)
            }

            try {
                os?.close()
            } catch (e: IOException) {
                Log.e(TAG, "Exception", e)
            }

            urlConnection?.disconnect()
        }
        return result
    }

    companion object {
        private const val TAG = "MobIdHttp"
        private const val HTTP_UNPROCESSABLE_ENTITY = 422
        private val URL ="https://api.mobid.ai"
    }
}