package com.mobid.android.http.repository

import android.annotation.SuppressLint
import android.util.Log
import com.mobid.android.http.RemoteError
import com.mobid.android.http.RemoteResult
import com.mobid.android.utils.getString
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import org.json.JSONTokener
import java.io.*
import java.net.*
import java.util.*
import javax.net.ssl.HttpsURLConnection
import javax.net.ssl.SSLHandshakeException

class HttpPostMultipart(
    private val token: String,
    private val charset: String = "UTF-8"
) {
    private var boundary: String = UUID.randomUUID().toString()
    private var httpConn: HttpsURLConnection? = null
    private var outputStream: OutputStream? = null
    private var writer: PrintWriter? = null

    /**
     * This constructor initializes a new HTTP POST request with content type
     * is set to multipart/form-data
     */

    @SuppressLint("LogNotTimber")
    suspend fun call(path: String, verificationId: String, status: String, file: File): RemoteResult<JSONObject> {
        var resource = RemoteResult<JSONObject>()
        try {
            httpConn =  URL(URL + path).openConnection() as HttpsURLConnection
            httpConn?.useCaches = false
            httpConn?.doOutput = true // indicates POST method
            httpConn?.doInput = true
            httpConn?.requestMethod = "POST"
            httpConn?.setRequestProperty("Content-Type", "multipart/form-data; boundary=$boundary")
            httpConn?.setRequestProperty("Accept", "application/json; indent=4")
            httpConn?.setRequestProperty("Authorization", "Bearer $token")
            outputStream = httpConn?.outputStream
            writer = PrintWriter(OutputStreamWriter(outputStream, charset), true)

            //
            addFormField("verification", verificationId)
            addFormField("type", status)
            addFilePart("file", file)
            //

            var outputJson = finish()
            val httpResponseCode = httpConn?.responseCode

            val json = JSONTokener(outputJson).nextValue()
            if (json is JSONArray) {
                outputJson = "{\"value\":$outputJson}"
            }

            if (httpResponseCode in listOf(
                            HttpURLConnection.HTTP_CREATED,
                            HttpURLConnection.HTTP_OK,
                            HttpURLConnection.HTTP_ACCEPTED
                    )
            ) {
                resource.data = JSONObject(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_NOT_FOUND) {
                resource.remoteError = RemoteError.HttpNotFound(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_GATEWAY_TIMEOUT) {
                resource.remoteError = RemoteError.HttpTimeout(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_BAD_GATEWAY) {
                resource.remoteError = RemoteError.HttpBadGetaway(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_FORBIDDEN) {
                resource.remoteError = RemoteError.HttpForbidden(outputJson)
            } else if (httpResponseCode == HttpURLConnection.HTTP_UNAUTHORIZED) {
                resource.remoteError = RemoteError.HttpUnauthorized(outputJson)
            } else {
                resource.remoteError = RemoteError.HttpUnauthorized(outputJson)
            }


        } catch (e: SSLHandshakeException) {
            Log.e(TAG, "Exception", e)
            resource.remoteError = RemoteError.SslException(null)
        } catch (e: JSONException) {
            Log.e(TAG, "Exception", e)
            resource.remoteError = RemoteError.JsonException(null)
        } catch (e: MalformedURLException) {
            Log.e(TAG, "Exception", e)
            resource.remoteError = RemoteError.MalformedUrlException(null)
        } catch (e: UnsupportedEncodingException) {
            Log.e(TAG, "Exception", e)
            resource.remoteError = RemoteError.UnsupportedEncodingException(null)
        } catch (e: ProtocolException) {
            Log.e(TAG, "Exception", e)
            resource.remoteError = RemoteError.NetProtocolException(null)
        } catch (e: SocketTimeoutException) {
            Log.e(TAG, "Exception", e)
            resource.remoteError = RemoteError.HttpTimeout(null)
        } catch (e: IOException) {
            Log.e(TAG, "Exception", e)
            resource.remoteError = RemoteError.NoInternetException(null)
        } catch (e: Exception) {
            Log.e(TAG, "Exception", e)
            resource.remoteError = RemoteError.Exception(null)
        } finally {
            try {
                writer?.close()
            } catch (e: IOException) {
                Log.e(TAG, "Exception", e)
            }

            try {
                outputStream?.close()
            } catch (e: IOException) {
                Log.e(TAG, "Exception", e)
            }

            httpConn?.disconnect()
        }
        return resource
    }

    /**
     * Adds a form field to the request
     *
     * @param name  field name
     * @param value field value
     */
    private fun addFormField(name: String, value: String?) {
        writer?.append("--$boundary")?.append(LINE)
        writer?.append("Content-Disposition: form-data; name=\"$name\"")
            ?.append(LINE)
        writer?.append("Content-Type: text/plain; charset=$charset")
            ?.append(LINE)
        writer?.append(LINE)
        writer?.append(value)?.append(LINE)
        writer?.flush()
    }

    /**
     * Adds a upload file section to the request
     *
     * @param fieldName
     * @param uploadFile
     * @throws IOException
     */
    @Throws(IOException::class)
    private fun addFilePart(fieldName: String, uploadFile: File) {
        var inputStream: InputStream? = null
        try{
            val fileName: String = uploadFile.name
            writer?.append("--$boundary")?.append(LINE)
            writer?.append("Content-Disposition: form-data; name=\"$fieldName\"; filename=\"$fileName\"")
                    ?.append(LINE)
            writer?.append("Content-Type: " + URLConnection.guessContentTypeFromName(fileName))
                    ?.append(LINE)
            writer?.append("Content-Transfer-Encoding: binary")?.append(LINE)
            writer?.append(LINE)
            writer?.flush()
            inputStream = FileInputStream(uploadFile)
            val buffer = ByteArray(4096)
            var bytesRead: Int
            while (inputStream.read(buffer).also { bytesRead = it } != -1) {
                outputStream?.write(buffer, 0, bytesRead)
            }
            outputStream?.flush()
            writer?.append(LINE)
            writer?.flush()
        }finally {
            inputStream?.close()
        }
    }

    /**
     * Completes the request and receives response from the server.
     *
     * @return String as response in case the server returned
     * status OK, otherwise an exception is thrown.
     * @throws IOException
     */
    @Throws(IOException::class)
    private fun finish(): String {
        writer?.flush()
        writer?.append("--$boundary--")?.append(LINE)
        writer?.close()

        // checks server's status code first
        var ins = try
        //temp fix for 401 and WWW-AUTH header bug
        {
            BufferedInputStream(httpConn?.inputStream)
        } catch (e: IOException) {
            BufferedInputStream(httpConn?.errorStream)
        }
        return ins.getString()
    }

    companion object {
        private val URL = "https://api.mobid.ai"
        private val TAG = "MobIdHttp"
        private const val LINE = "\r\n"
    }


}