package com.mobid.android.view.camera

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.provider.MediaStore
import android.view.View
import androidx.appcompat.app.AppCompatActivity
import androidx.core.net.toFile
import androidx.fragment.app.Fragment
import androidx.lifecycle.coroutineScope
import com.mobid.android.R
import com.mobid.android.http.isSuccess
import com.mobid.android.http.repository.MobIdRepository
import com.mobid.android.model.ConferenceStatus
import com.mobid.android.utils.showError
import com.mobid.android.utils.toRealFile
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.io.File

internal class CameraActivity: AppCompatActivity(), CameraFragmentBridge{

    private var currentFragment: Fragment? = null
    private val repo by lazy { MobIdRepository.getInstance() }
    private val progressLayout by lazy { findViewById<View>(R.id.progressLayout) }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_camera)

        currentFragment = CameraFragment.getInstance()
        supportFragmentManager
            .beginTransaction()
            .add(R.id.fragmentContainer, currentFragment!!)
            .commit()
    }

    override fun onNewPicture(uri: Uri) {
        currentFragment = PhotoPreviewFragment.getInstance(uri)
        supportFragmentManager
                .beginTransaction()
                .replace(R.id.fragmentContainer, currentFragment!!, "")
                .commit()
    }

    override fun openPhotoList() {
        val pickImageIntent = Intent(
            Intent.ACTION_PICK,
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI)

        startActivityForResult(pickImageIntent, REQUEST_CODE_GALLERY)
    }


    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        if(requestCode == REQUEST_CODE_GALLERY && resultCode == Activity.RESULT_OK){
            data?.data?.let {
                onNewPicture(it)
            }
        }
    }

    override fun closePhotoList() {
        onBackPressed()
    }

    override fun goBack() {
        onBackPressed()
    }

    override fun onComplete(uri: Uri) {
        lifecycle.coroutineScope.launch(Dispatchers.IO){
            progressLayout.post { progressLayout.visibility = View.VISIBLE }
            val file = uri
                    .toRealFile(this@CameraActivity,File(cacheDir, provideFileName()))
                    .toFile()
            val result = repo.uploadFile(
                    verificationId = provideConferenceId(),
                    type = provideStatus(),
                    file = file,
                    token = provideToken()
            )

            when{
                result.isSuccess() ->{
                    file.delete()
                    progressLayout.post { progressLayout.visibility = View.GONE }
                    setResult(Activity.RESULT_OK, Intent().setData(uri))
                    finish()
                }
                else -> {
                    lifecycle.coroutineScope.launch(Dispatchers.Main) {
                        showError(
                            toastMessage = result.remoteError?.message?:getString(R.string.error_upload_image),
                            errorLog = "Error with multipart")
                    }
                }
            }
        }
    }

    private fun provideFileName() = "${provideStatus().id}_${provideConferenceId()}.jpeg"

    companion object {

        private const val REQUEST_CODE_GALLERY = 1212
        private const val EXTRA_CONFERENCE_ID = "EXTRA_CONFERENCE_ID"
        private const val EXTRA_STATUS = "EXTRA_STATUS"
        private const val EXTRA_TOKEN = "EXTRA_TOKEN"

        private fun CameraActivity.provideConferenceId() =
            intent.getStringExtra(EXTRA_CONFERENCE_ID)?:""

        private fun CameraActivity.provideStatus() =
            intent.getSerializableExtra(EXTRA_STATUS) as ConferenceStatus

        private fun CameraActivity.provideToken() =
            intent.getStringExtra(EXTRA_TOKEN)?:""

        fun getLauncher(context: Context, verificationId: String, conferenceStatus: ConferenceStatus, token: String) =
            Intent(context, CameraActivity::class.java).apply {
                putExtra(EXTRA_CONFERENCE_ID, verificationId)
                putExtra(EXTRA_STATUS, conferenceStatus)
                putExtra(EXTRA_TOKEN, token)
            }
    }
}

