package com.mobid.android.view.camera

import android.Manifest
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.fragment.app.Fragment
import androidx.lifecycle.coroutineScope
import com.google.android.material.snackbar.Snackbar
import com.mobid.android.R
import com.mobid.android.utils.isGrantedCameraPerm
import com.mobid.android.utils.isGrantedStoragePerm
import com.mobid.android.utils.showError
import io.fotoapparat.Fotoapparat
import io.fotoapparat.configuration.CameraConfiguration
import io.fotoapparat.parameter.ScaleType
import io.fotoapparat.selector.back
import io.fotoapparat.selector.front
import io.fotoapparat.view.CameraView
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.io.File

internal class CameraFragment: Fragment(){

    private var fragmentBridge: CameraFragmentBridge? = null
    private val cameraView by lazy { view?.findViewById<CameraView>(R.id.cameraView) }
    private var fotoapparat: Fotoapparat? = null
    private var isFront = true


    override fun onAttach(context: Context) {
        super.onAttach(context)
        fragmentBridge = context as CameraFragmentBridge
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {

        return inflater.inflate(R.layout.fragment_camera, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        checkPermission()
        view.findViewById<View>(R.id.imageButtonCancel).setOnClickListener {
            fragmentBridge?.goBack()
        }
        view.findViewById<View>(R.id.imageButtonGallery).setOnClickListener {
            checkStoragePermission()
        }
        view.findViewById<View>(R.id.imageButtonShoot).setOnClickListener {
            takePhoto()
        }
        view.findViewById<View>(R.id.imageButtonChangeCamera).setOnClickListener {
            isFront = !isFront
            fotoapparat?.switchTo(
                    lensPosition = if(isFront) front() else back(),
                    cameraConfiguration = CameraConfiguration.standard()
            )
        }
    }

    private fun checkStoragePermission(){
        if(requireContext().isGrantedStoragePerm()) {
            fragmentBridge?.openPhotoList()
        }else {
            requestPermissions(arrayOf(GALLERY_PERMISSION), REQUEST_CODE_STORAGE_PERMISSION)
        }
    }

    private fun checkPermission(){
        if(requireContext().isGrantedCameraPerm()){
            initFotoaparat()
        }else{
            requestPermissions(arrayOf(CAMERA_PERMISSION), REQUEST_CODE_CAMERA_PERMISSION)
        }
    }

    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        if(requestCode == REQUEST_CODE_CAMERA_PERMISSION ){
            if(requireContext().isGrantedCameraPerm()){
                initFotoaparat()
            }else{
                showError(
                        toastMessage = getString(R.string.error_permission_camera),
                        errorLog = "Manifest.permission.CAMERA not granted"
                )
            }

        } else if(requestCode == REQUEST_CODE_STORAGE_PERMISSION){
            if(requireContext().isGrantedStoragePerm())
                fragmentBridge?.openPhotoList()
            else
               showError(
                        toastMessage = getString(R.string.error_permission_gallery),
                        errorLog = "Manifest.permission.READ_EXTERNAL_STORAGE not granted"
                )
        }
    }

    private fun initFotoaparat(){
        cameraView?:return
        fotoapparat = Fotoapparat(
            context = requireContext(),
            view = cameraView!!,
            scaleType = ScaleType.CenterCrop,
            lensPosition = if(isFront) front() else back(),
            cameraErrorCallback = {onCameraError(it)}
        )
    }

    override fun onStart() {
        super.onStart()
        fotoapparat?.start()
    }

    override fun onStop() {
        super.onStop()
        fotoapparat?.stop()
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        if(requestCode == REQUEST_CODE_GALLERY && resultCode == Activity.RESULT_OK){
            if(data?.data != null)
                fragmentBridge?.onNewPicture(data.data!!)
            else
                onCameraError(Throwable(getString(R.string.error_from_gallery)))
        }
    }

    private fun takePhoto(){
        lifecycle.coroutineScope.launch (Dispatchers.IO){
            val image = File(requireContext().cacheDir, "photo1.jpeg")
            fotoapparat?.takePicture()?.saveToFile(image)?.await()
            val uri = Uri.fromFile(image)
            lifecycle.coroutineScope.launch (Dispatchers.Main){
                fragmentBridge?.onNewPicture(uri)
            }
        }

    }


    private fun onCameraError(error: Throwable){
        error.printStackTrace()
        cameraView?:return
        val errorMessage = error.localizedMessage?:getString(R.string.error_camera)
        Snackbar.make(cameraView!!, errorMessage, Snackbar.LENGTH_LONG).show()

    }

    companion object {
        private const val REQUEST_CODE_CAMERA_PERMISSION = 1210
        private const val REQUEST_CODE_STORAGE_PERMISSION = 1211
        private const val REQUEST_CODE_GALLERY = 1212
        private const val CAMERA_PERMISSION = Manifest.permission.CAMERA
        private const val GALLERY_PERMISSION = Manifest.permission.READ_EXTERNAL_STORAGE

        fun getInstance() = CameraFragment().apply {
            arguments = Bundle()
        }
    }
}
