package ai.mrs.connection;

import ai.mrs.AbstractMasonsSDK;
import ai.mrs.connection.models.UtteranceResponse;
import ai.mrs.connection.utils.HTTPUtil;
import ai.mrs.mdp.MDPClient;
import ai.mrs.mdp.MDPHandler;
import ai.mrs.session.CalleeSession;
import ai.mrs.session.CallerSession;
import ai.mrs.session.models.CreatingSessionOfCallee;
import ai.mrs.MasonsSDKConfig;
import ai.mrs.session.models.ReplyFromCallee;
import ai.mrs.session.models.UtteranceFromCaller;
import ai.mrs.utils.JsonUtil;

import java.util.HashMap;
import java.util.Map;

public class MasonsConnection implements MDPHandler {
    private String nodeID;
    private MDPClient mdpClient;
    private AbstractMasonsSDK sdk;
    private MasonsSDKConfig config;

    public MasonsConnection(AbstractMasonsSDK sdk, MasonsSDKConfig config) {
        this.config = config;
        this.sdk = sdk;
    }

    private void createNode() {
        String nodeUrl = this.config.getNodeUrl();
        Map<String, String> headers = new HashMap<>();
        headers.put("Access-Token", this.config.getAgentToken());
        String resp = HTTPUtil.post(nodeUrl, "", headers);
        Map respMap = JsonUtil.jsonToMap(resp);
        this.nodeID = (String) respMap.get("node_id");
    }

    public void start() {
        this.createNode();
        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Sec-WebSocket-Protocol", "Duplex");
        headers.put("Access-Token", config.getAgentToken());
        headers.put("Node-ID", this.nodeID);

        this.mdpClient = new MDPClient(
                this.config.getWsUrl(), this, headers,
                this.config.getConnectTimeout(), this.config.getRpcTimeout()
        );
        this.mdpClient.connect();
    }

    public void stop() {
        this.mdpClient.close();
    }

    public Map callKnockEvent(Map data) {
        return this.mdpClient.callRPC("knock", data);
    }

    public UtteranceResponse callUtteranceEvent(String sessionID, String text) {
        Map<String, Object> req = new HashMap<>();
        req.put("session_id", sessionID);
        req.put("text", text);

        Map resp = this.mdpClient.callRPC("utterance", req);
        return JsonUtil.mapToObj(resp, UtteranceResponse.class);
    }

    public void sendReplyEvent(String sessionID, String text, boolean isEnd) {
        Map<String, Object> data = new HashMap<>();
        data.put("session_id", sessionID);
        data.put("text", text);
        data.put("isEnd", isEnd);
        this.mdpClient.sendEvent("reply", data);
    }

    @Override
    public void processEventMessage(String event, Map data) {
        String sessionID = (String) data.get("session_id");

        switch (event) {
            case "utterance":
                String text = (String) data.get("text");
                CallerSession callerSession = this.sdk.getCallerSessionBySessionID(sessionID);
                UtteranceFromCaller utterance = new UtteranceFromCaller(callerSession, text);
                this.sdk.onReceivingUtteranceFromCaller(utterance);
                break;
            case "reply":
                ReplyFromCallee reply = new ReplyFromCallee();
                this.sdk.onReceivingReplyFromCallee(reply);
                break;
            case "create": {
                String accountKey = (String) data.get("account_key");
                this.sdk.createCalleeSession(sessionID, accountKey);

                CreatingSessionOfCallee sessionWrapper = new CreatingSessionOfCallee();
                sessionWrapper.setAccountKey(accountKey);

                this.sdk.onCreatingSessionOfCallee(sessionWrapper);
                break;
            }
            case "exit": {
                CalleeSession session = this.sdk.getCalleeSessionBySessionID(sessionID);
                String accountKey = session.getAccountKey();
                this.sdk.removeCalleeSession(sessionID, accountKey);

                CreatingSessionOfCallee sessionWrapper = new CreatingSessionOfCallee();
                sessionWrapper.setAccountKey(accountKey);

                this.sdk.onCreatingSessionOfCallee(sessionWrapper);
                break;
            }
            default:
                this.mdpClient.sendError("This event is not supported");
        }
    }

    @Override
    public Map processRPCRequest(String event, Map data) {
        return new HashMap();
    }
}
