package ai.nextbillion.jacoco

import android.os.Bundle
import android.content.Intent
import android.os.Looper
import android.app.Activity
import android.app.Instrumentation
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.io.OutputStream
import java.lang.Exception

/**
 * @author lihongjun
 * @version 1.0
 * @since 2023/9/5
 */
class JacocoInstrumentation : Instrumentation(), FinishListener {

    private val mResults = Bundle()
    private var mIntent: Intent? = null
    private val mCoverage = true
    private var mCoverageFilePath: String? = null
    override fun onCreate(arguments: Bundle) {
        LogUtil.e(TAG, "onCreate($arguments)")
        super.onCreate(arguments)
        DEFAULT_COVERAGE_FILE_PATH = context.filesDir.path + "/coverage.ec"
        val file = File(DEFAULT_COVERAGE_FILE_PATH)
        if (file.isFile && file.exists()) {
            if (file.delete()) {
                LogUtil.e(TAG, "file del successs")
            } else {
                LogUtil.e(TAG, "file del fail !")
            }
        }
        if (!file.exists()) {
            try {
                file.createNewFile()
            } catch (e: IOException) {
                LogUtil.e(TAG, "异常 : $e")
                e.printStackTrace()
            }
        }
        if (arguments != null) {
            LogUtil.e(TAG, "arguments不为空 : $arguments")
            mCoverageFilePath = arguments.getString("coverageFile")
            LogUtil.e(TAG, "mCoverageFilePath = $mCoverageFilePath")
        }
        mIntent = Intent(targetContext, InstrumentedActivity::class.java)
        mIntent!!.flags = Intent.FLAG_ACTIVITY_NEW_TASK
        start()
    }

    override fun onStart() {
        LogUtil.e(TAG, "onStart def")
        if (LOGD) {
            LogUtil.e(TAG, "onStart()")
        }
        super.onStart()
        Looper.prepare()
        val activity = startActivitySync(mIntent) as InstrumentedActivity
        activity.setFinishListener(this)
    }

    private fun getBooleanArgument(arguments: Bundle, tag: String): Boolean {
        val tagString = arguments.getString(tag)
        return tagString != null && java.lang.Boolean.parseBoolean(tagString)
    }

    private fun generateCoverageReport() {
        var out: OutputStream? = null
        try {
            out = FileOutputStream(coverageFilePath, false)
            val agent = Class.forName("org.jacoco.agent.rt.RT")
                .getMethod("getAgent")
                .invoke(null)
            out.write(
                agent.javaClass.getMethod("getExecutionData", Boolean::class.javaPrimitiveType)
                    .invoke(agent, false) as ByteArray
            )
        } catch (e: Exception) {
            LogUtil.e(TAG, e.toString())
            e.printStackTrace()
        } finally {
            if (out != null) {
                try {
                    out.close()
                } catch (e: IOException) {
                    e.printStackTrace()
                }
            }
        }
    }

    private val coverageFilePath: String
        private get() = mCoverageFilePath ?: DEFAULT_COVERAGE_FILE_PATH

    private fun setCoverageFilePath(filePath: String?): Boolean {
        if (filePath != null && filePath.length > 0) {
            mCoverageFilePath = filePath
            return true
        }
        return false
    }

    private fun reportEmmaError(e: Exception) {
        reportEmmaError("", e)
    }

    private fun reportEmmaError(hint: String, e: Exception) {
        val msg = "Failed to generate emma coverage. $hint"
        LogUtil.e(TAG, msg)
        mResults.putString(
            REPORT_KEY_STREAMRESULT, """
     
     Error: $msg
     """.trimIndent()
        )
    }

    override fun onActivityFinished() {
        if (LOGD) {
            LogUtil.e(TAG, "onActivityFinished()")
        }
        if (mCoverage) {
            LogUtil.e(TAG, "onActivityFinished mCoverage true")
            generateCoverageReport()
        }
        finish(Activity.RESULT_OK, mResults)
    }

    override fun dumpIntermediateCoverage(filePath: String?) {

        // TODO Auto-generated method stub
        if (LOGD) {
            LogUtil.e(TAG, "Intermidate Dump Called with file name :$filePath")
        }
        if (mCoverage) {
            if (!setCoverageFilePath(filePath)) {
                if (LOGD) {
                    LogUtil.e(TAG, "Unable to set the given file path:$filePath as dump target.")
                }
            }
            generateCoverageReport()
            setCoverageFilePath(DEFAULT_COVERAGE_FILE_PATH)
        }
    }

    companion object {
        var TAG = "JacocoInstrumentation:"
        private var DEFAULT_COVERAGE_FILE_PATH = "/mnt/sdcard/coverage.ec"
        private const val LOGD = true
    }
}