package ai.nextbillion.maps;

import ai.nextbillion.maps.errors.ApiException;
import ai.nextbillion.maps.internal.ApiConfig;
import ai.nextbillion.maps.internal.ApiResponse;
import ai.nextbillion.maps.internal.StringJoin;
import ai.nextbillion.maps.model.LatLng;
import ai.nextbillion.maps.model.SnappedPoint;
import com.google.gson.FieldNamingPolicy;

/**
 * The Google Maps Roads API identifies the roads a vehicle was traveling along and provides
 * additional metadata about those roads, such as speed limits.
 *
 * <p>See also: <a href="https://developers.google.com/maps/documentation/roads">Roads API
 * documentation</a>.
 */
public class RoadsApi {

  static final ApiConfig SNAP_TO_ROADS_API_CONFIG =
      new ApiConfig("/snapToRoads/json").fieldNamingPolicy(FieldNamingPolicy.IDENTITY);

  private RoadsApi() {}

  /**
   * Takes up to 100 GPS points collected along a route, and returns a similar set of data with the
   * points snapped to the most likely roads the vehicle was traveling along.
   *
   * @param context The {@link GeoApiContext} to make requests through.
   * @param path The collected GPS points as a path.
   * @return Returns the snapped points as a {@link PendingResult}.
   */
  public static PendingResult<SnappedPoint[]> snapToRoads(GeoApiContext context, LatLng... path) {
    return context.get(
        SNAP_TO_ROADS_API_CONFIG, RoadsResponse.class, "path", StringJoin.join('|', path));
  }

  public static class RoadsResponse implements ApiResponse<SnappedPoint[]> {
    public String status;
    public SnappedPoint[] snappedPoints;
    public double distance;
    public String errorMessage;

    @Override
    public boolean successful() {
      return "Ok".equals(status);
    }

    @Override
    public SnappedPoint[] getResult() {
      return snappedPoints;
    }

    @Override
    public ApiException getError() {
      if (successful()) {
        return null;
      }
      return ApiException.from(status, errorMessage);
    }
  }
}
