package com.nbmap.api.directions.v5;

import com.nbmap.api.directions.v5.models.DirectionsResponse;
import com.nbmap.api.directions.v5.models.DirectionsRoute;
import com.nbmap.api.directions.v5.models.RouteOptions;
import com.nbmap.api.directions.v5.utils.ParseUtils;

import java.util.ArrayList;
import java.util.List;

import retrofit2.Response;

/**
 * @since 4.4.0
 */
class DirectionsResponseFactory {

  private final NbmapDirections nbmapDirections;

  DirectionsResponseFactory(NbmapDirections nbmapDirections) {
    this.nbmapDirections = nbmapDirections;
  }

  Response<DirectionsResponse> generate(Response<DirectionsResponse> response) {
    if (isNotSuccessful(response)) {
      return response;
    } else {
      return Response.success(
        response
          .body()
          .toBuilder()
          .routes(generateRouteOptions(response))
          .build(),
        new okhttp3.Response.Builder()
          .code(200)
          .message("OK")
          .protocol(response.raw().protocol())
          .headers(response.headers())
          .request(response.raw().request())
          .build());
    }
  }

  private boolean isNotSuccessful(Response<DirectionsResponse> response) {
    return !response.isSuccessful()
      || response.body() == null
      || response.body().routes().isEmpty();
  }

  private List<DirectionsRoute> generateRouteOptions(Response<DirectionsResponse> response) {
    List<DirectionsRoute> routes = response.body().routes();
    List<DirectionsRoute> modifiedRoutes = new ArrayList<>();
    for (DirectionsRoute route : routes) {
      modifiedRoutes.add(route.toBuilder().routeOptions(
        RouteOptions.builder()
          .profile(nbmapDirections.profile())
          .coordinates(nbmapDirections.coordinates())
          .waypointIndicesList(ParseUtils.parseToIntegers(nbmapDirections.waypointIndices()))
          .waypointNamesList(ParseUtils.parseToStrings(nbmapDirections.waypointNames()))
          .waypointTargetsList(ParseUtils.parseToPoints(nbmapDirections.waypointTargets()))
          .continueStraight(nbmapDirections.continueStraight())
          .annotationsList(ParseUtils.parseToStrings(nbmapDirections.annotation()))
          .approachesList(ParseUtils.parseToStrings(nbmapDirections.approaches()))
          .bearingsList(ParseUtils.parseToListOfListOfDoubles(nbmapDirections.bearing()))
          .alternatives(nbmapDirections.alternatives())
          .language(nbmapDirections.language())
          .radiusesList(ParseUtils.parseToDoubles(nbmapDirections.radius()))
          .user(nbmapDirections.user())
          .voiceInstructions(nbmapDirections.voiceInstructions())
          .bannerInstructions(nbmapDirections.bannerInstructions())
          .roundaboutExits(nbmapDirections.roundaboutExits())
          .geometries(nbmapDirections.geometries())
          .overview(nbmapDirections.overview())
          .steps(nbmapDirections.steps())
          .exclude(nbmapDirections.exclude())
          .voiceUnits(nbmapDirections.voiceUnits())
          .accessToken(nbmapDirections.accessToken())
          .requestUuid(response.body().uuid())
          .baseUrl(nbmapDirections.baseUrl())
          .walkingOptions(nbmapDirections.walkingOptions())
          .snappingClosures(nbmapDirections.snappingClosures())
          .build()
      ).build());
    }
    return modifiedRoutes;
  }
}
