package com.nbmap.api.matching.v5;

import com.nbmap.api.directions.v5.models.RouteOptions;
import com.nbmap.api.directions.v5.utils.ParseUtils;
import com.nbmap.api.matching.v5.models.MapMatchingMatching;
import com.nbmap.api.matching.v5.models.MapMatchingResponse;
import com.nbmap.geojson.Point;

import java.util.ArrayList;
import java.util.List;

import retrofit2.Response;

/**
 * @since 4.4.0
 */
class MatchingResponseFactory {

  private static final String PLACEHOLDER_UUID = "mapmatching";
  private final NbmapMapMatching nbmapMapMatching;

  MatchingResponseFactory(NbmapMapMatching nbmapMapMatching) {
    this.nbmapMapMatching = nbmapMapMatching;
  }

  Response<MapMatchingResponse> generate(Response<MapMatchingResponse> response) {
    if (isNotSuccessful(response)) {
      return response;
    } else {
      return Response.success(
        response
          .body()
          .toBuilder()
          .matchings(generateRouteOptions(response))
          .build(),
        new okhttp3.Response.Builder()
          .code(200)
          .message("OK")
          .protocol(response.raw().protocol())
          .headers(response.headers())
          .request(response.raw().request())
          .build());
    }
  }

  private boolean isNotSuccessful(Response<MapMatchingResponse> response) {
    return !response.isSuccessful()
      || response.body() == null
      || response.body().matchings() == null
      || response.body().matchings().isEmpty();
  }

  private List<MapMatchingMatching> generateRouteOptions(Response<MapMatchingResponse> response) {
    List<MapMatchingMatching> matchings = response.body().matchings();
    List<MapMatchingMatching> modifiedMatchings = new ArrayList<>();
    for (MapMatchingMatching matching : matchings) {
      modifiedMatchings.add(matching.toBuilder().routeOptions(
        RouteOptions.builder()
          .profile(nbmapMapMatching.profile())
          .coordinates(formatCoordinates(nbmapMapMatching.coordinates()))
          .annotations(nbmapMapMatching.annotations())
          .approachesList(ParseUtils.parseToStrings(nbmapMapMatching.approaches()))
          .language(nbmapMapMatching.language())
          .radiusesList(ParseUtils.parseToDoubles(nbmapMapMatching.radiuses()))
          .user(nbmapMapMatching.user())
          .voiceInstructions(nbmapMapMatching.voiceInstructions())
          .bannerInstructions(nbmapMapMatching.bannerInstructions())
          .roundaboutExits(nbmapMapMatching.roundaboutExits())
          .geometries(nbmapMapMatching.geometries())
          .overview(nbmapMapMatching.overview())
          .steps(nbmapMapMatching.steps())
          .voiceUnits(nbmapMapMatching.voiceUnits())
          .requestUuid(PLACEHOLDER_UUID)
          .accessToken(nbmapMapMatching.accessToken())
          .waypointIndicesList(ParseUtils.parseToIntegers(nbmapMapMatching.waypointIndices()))
          .waypointNamesList(ParseUtils.parseToStrings(nbmapMapMatching.waypointNames()))
          .baseUrl(nbmapMapMatching.baseUrl())
          .build()
      ).build());
    }
    return modifiedMatchings;
  }

  private static List<Point> formatCoordinates(String coordinates) {
    String[] coordPairs = coordinates.split(";", -1);
    List<Point> coordinatesFormatted = new ArrayList<>();
    for (String coordPair : coordPairs) {
      String[] coords = coordPair.split(",", -1);
      coordinatesFormatted.add(
        Point.fromLngLat(Double.valueOf(coords[0]), Double.valueOf(coords[1])));

    }
    return coordinatesFormatted;
  }
}
