package com.nbmap.api.routetiles.v1.versions;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.auto.value.AutoValue;
import com.google.gson.GsonBuilder;
import com.nbmap.api.routetiles.v1.NbmapRouteTiles;
import com.nbmap.api.routetiles.v1.versions.models.RouteTileVersionsAdapterFactory;
import com.nbmap.api.routetiles.v1.versions.models.RouteTileVersionsResponse;
import com.nbmap.core.NbmapService;
import com.nbmap.core.constants.Constants;
import com.nbmap.core.exceptions.ServicesException;
import com.nbmap.core.utils.ApiCallHelper;
import com.nbmap.core.utils.NbmapUtils;

import retrofit2.Call;

/**
 * The Route Tile Versions API allows the fetching of all available versions of route tiles
 * currently available. It is used in conjunction with the {@link NbmapRouteTiles} API.
 *
 * @since 4.1.0
 */
@AutoValue
public abstract class NbmapRouteTileVersions extends NbmapService<RouteTileVersionsResponse,
  RouteTileVersionsService> {

  protected NbmapRouteTileVersions() {
    super(RouteTileVersionsService.class);
  }

  @Override
  protected GsonBuilder getGsonBuilder() {
    return new GsonBuilder()
      .registerTypeAdapterFactory(RouteTileVersionsAdapterFactory.create());
  }

  @Override
  protected Call<RouteTileVersionsResponse> initializeCall() {
    return getService().getCall(
      ApiCallHelper.getHeaderUserAgent(clientAppName()),
      accessToken()
    );
  }

  @Nullable
  abstract String clientAppName();

  @NonNull
  abstract String accessToken();

  @Override
  protected abstract String baseUrl();

  /**
   * Build a new {@link NbmapRouteTileVersions} object.
   *
   * @return a {@link Builder} object for creating this object
   * @since 4.1.0
   */
  public static Builder builder() {
    return new AutoValue_NbmapRouteTileVersions.Builder()
      .baseUrl(Constants.BASE_API_URL);
  }

  /**
   * Returns the builder which created this instance of {@link NbmapRouteTileVersions} and
   * allows for modification and building a new route tile versions request with new information.
   *
   * @return {@link Builder} with the same variables set as this route tile versions object
   * @since 4.1.0
   */
  public abstract Builder toBuilder();

  /**
   * This builder is used to create a new request to the Nbmap Route Tiles API. At a bare minimum,
   * your request must include an access token.
   *
   * @since 4.1.0
   */
  @AutoValue.Builder
  public abstract static class Builder {

    /**
     * Required to call when this is being built. If no access token provided,
     * {@link ServicesException} will be thrown.
     *
     * @param accessToken Nbmap access token, You must have a Nbmap account inorder to use
     *                    the Route Tiles API
     * @return this builder for chaining options together
     * @since 4.1.0
     */
    public abstract Builder accessToken(@NonNull String accessToken);

    /**
     * Optionally change the APIs base URL to something other then the default Nbmap one.
     *
     * @param baseUrl base url used as end point
     * @return this builder for chaining options together
     * @since 4.1.0
     */
    public abstract Builder baseUrl(@NonNull String baseUrl);

    /**
     * Base package name or other simple string identifier. Used inside the calls user agent header.
     *
     * @param clientAppName base package name or other simple string identifier
     * @return this builder for chaining options together
     * @since 4.1.0
     */
    public abstract Builder clientAppName(@NonNull String clientAppName);

    abstract NbmapRouteTileVersions autoBuild();

    /**
     * This uses the provided parameters set using the {@link Builder} and first checks that all
     * values are valid, and creates a new {@link NbmapRouteTileVersions} object with the values
     * provided.
     *
     * @return a new instance of Nbmap Route Tiles Version
     * @throws ServicesException when a provided parameter is detected to be incorrect
     * @since 4.1.0
     */
    public NbmapRouteTileVersions build() {
      NbmapRouteTileVersions nbmapRouteTileVersions = autoBuild();

      if (!NbmapUtils.isAccessTokenValid(nbmapRouteTileVersions.accessToken())) {
        throw new ServicesException("Using Nbmap Services requires setting a valid access token.");
      }

      return nbmapRouteTileVersions;
    }
  }
}
