/*
 * Decompiled with CFR 0.152.
 */
package ai.nightfall.scan;

import ai.nightfall.scan.model.BaseNightfallException;
import ai.nightfall.scan.model.CompleteFileUploadRequest;
import ai.nightfall.scan.model.FileUpload;
import ai.nightfall.scan.model.InitializeFileUploadRequest;
import ai.nightfall.scan.model.NightfallAPIException;
import ai.nightfall.scan.model.NightfallClientException;
import ai.nightfall.scan.model.NightfallErrorResponse;
import ai.nightfall.scan.model.NightfallRequestTimeoutException;
import ai.nightfall.scan.model.ScanFileRequest;
import ai.nightfall.scan.model.ScanFileResponse;
import ai.nightfall.scan.model.ScanTextRequest;
import ai.nightfall.scan.model.ScanTextResponse;
import ai.nightfall.scan.model.UploadFileChunkRequest;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import okhttp3.Call;
import okhttp3.ConnectionPool;
import okhttp3.Headers;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class NightfallClient
implements Closeable {
    private static final ObjectMapper objectMapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    private static final long wakeupDurationMillis = Duration.ofSeconds(15L).toMillis();
    private static final int DEFAULT_RETRY_COUNT = 5;
    private static final String API_HOST = "https://api.nightfall.ai";
    private final String implVersion = this.loadImplVersion();
    private final String apiHost;
    private final String apiKey;
    private final int fileUploadConcurrency;
    private final int retryCount;
    private final ExecutorService executor;
    private final OkHttpClient httpClient;

    NightfallClient(String apiHost, String apiKey, int fileUploadConcurrency, OkHttpClient httpClient) {
        this.apiHost = apiHost;
        this.apiKey = apiKey;
        this.fileUploadConcurrency = fileUploadConcurrency;
        this.retryCount = 5;
        this.executor = Executors.newFixedThreadPool(this.fileUploadConcurrency);
        this.httpClient = httpClient;
    }

    private String loadImplVersion() {
        Package pkg = this.getClass().getPackage();
        if (pkg == null) {
            return "";
        }
        return pkg.getImplementationVersion();
    }

    @Override
    public void close() {
        this.executor.shutdown();
        this.httpClient.dispatcher().executorService().shutdown();
    }

    public ScanTextResponse scanText(ScanTextRequest request) {
        byte[] jsonBody;
        if (request == null) {
            throw new IllegalArgumentException("request must be non-null");
        }
        try {
            jsonBody = objectMapper.writeValueAsBytes((Object)request);
        }
        catch (JsonProcessingException e) {
            throw new NightfallClientException("processing scan request: " + e.getMessage());
        }
        MediaType json = MediaType.parse((String)"application/json");
        return this.issueRequest("/v3/scan", "POST", json, jsonBody, null, ScanTextResponse.class);
    }

    public ScanFileResponse scanFile(ScanFileRequest request, InputStream content, long contentSizeBytes) {
        return this.scanFile(request, content, contentSizeBytes, null);
    }

    public ScanFileResponse scanFile(ScanFileRequest request, InputStream content, long contentSizeBytes, Duration timeout) {
        AtomicReference<BaseNightfallException> uploadException;
        InitializeFileUploadRequest initRequest;
        FileUpload upload;
        boolean uploadSuccess;
        if (request == null) {
            throw new IllegalArgumentException("request must be non-null");
        }
        if (content == null) {
            throw new IllegalArgumentException("content must be non-null");
        }
        Instant deadline = null;
        if (timeout != null) {
            if (timeout.isNegative()) {
                throw new IllegalArgumentException("timeout must be positive");
            }
            deadline = Instant.now().plus(timeout);
        }
        if (!(uploadSuccess = this.doChunkedUpload(upload = this.initializeFileUpload(initRequest = new InitializeFileUploadRequest(contentSizeBytes)), content, deadline, uploadException = new AtomicReference<BaseNightfallException>()))) {
            BaseNightfallException except = uploadException.get();
            if (except != null) {
                throw except;
            }
            throw new NightfallClientException("internal error: failed to upload all chunks of file");
        }
        CompleteFileUploadRequest completeReq = new CompleteFileUploadRequest(upload.getFileID());
        upload = this.completeFileUpload(completeReq);
        return this.scanUploadedFile(request, upload.getFileID());
    }

    private boolean doChunkedUpload(FileUpload upload, InputStream content, Instant deadline, AtomicReference<BaseNightfallException> uploadException) {
        int numPermits = this.fileUploadConcurrency;
        Semaphore semaphore = new Semaphore(numPermits);
        AtomicBoolean allChunksSucceed = new AtomicBoolean(true);
        int offset = 0;
        while ((long)offset < upload.getFileSizeBytes()) {
            semaphore.acquireUninterruptibly();
            this.checkFileUploadDeadline(deadline);
            if (!allChunksSucceed.get()) {
                return false;
            }
            UploadFileChunkRequest chunkReq = new UploadFileChunkRequest(upload.getFileID(), offset);
            byte[] data = new byte[(int)upload.getChunkSize()];
            try {
                boolean notLastChunk;
                int bytesRead = content.read(data);
                boolean bl = notLastChunk = (long)offset + upload.getChunkSize() < upload.getFileSizeBytes();
                if (bytesRead < data.length && notLastChunk) {
                    semaphore.release();
                    throw new NightfallClientException("failed to read data from input stream");
                }
                if (bytesRead < data.length) {
                    data = Arrays.copyOfRange(data, 0, bytesRead);
                }
            }
            catch (IOException e) {
                semaphore.release();
                throw new NightfallClientException("reading content to upload: " + e.getMessage());
            }
            chunkReq.setContent(data);
            this.executor.execute(() -> {
                try {
                    this.uploadFileChunk(chunkReq);
                }
                catch (BaseNightfallException e) {
                    allChunksSucceed.set(false);
                    uploadException.set(e);
                }
                catch (Throwable t) {
                    allChunksSucceed.set(false);
                }
                finally {
                    semaphore.release();
                }
            });
            offset = (int)((long)offset + upload.getChunkSize());
        }
        try {
            while (true) {
                boolean success;
                if (success = semaphore.tryAcquire(numPermits, wakeupDurationMillis, TimeUnit.MILLISECONDS)) {
                    return allChunksSucceed.get();
                }
                this.checkFileUploadDeadline(deadline);
            }
        }
        catch (InterruptedException e) {
            throw new NightfallClientException("interrupted while waiting for upload to complete");
        }
    }

    private void checkFileUploadDeadline(Instant deadline) {
        if (deadline != null && Instant.now().isAfter(deadline)) {
            throw new NightfallRequestTimeoutException("timed out while uploading file");
        }
    }

    private FileUpload initializeFileUpload(InitializeFileUploadRequest request) {
        byte[] jsonBody;
        try {
            jsonBody = objectMapper.writeValueAsBytes((Object)request);
        }
        catch (JsonProcessingException e) {
            throw new NightfallClientException("processing init-upload request: " + e.getMessage());
        }
        MediaType json = MediaType.parse((String)"application/json");
        return this.issueRequest("/v3/upload", "POST", json, jsonBody, null, FileUpload.class);
    }

    private boolean uploadFileChunk(UploadFileChunkRequest request) {
        Headers headers = Headers.of((String[])new String[]{"X-Upload-Offset", Long.toString(request.getFileOffset())});
        String path = "/v3/upload/" + request.getFileUploadID().toString();
        MediaType octetStream = MediaType.parse((String)"application/octet-stream");
        this.issueRequest(path, "PATCH", octetStream, request.getContent(), headers, Void.class);
        return true;
    }

    private FileUpload completeFileUpload(CompleteFileUploadRequest request) {
        String path = "/v3/upload/" + request.getFileUploadID().toString() + "/finish";
        MediaType json = MediaType.parse((String)"application/json");
        return this.issueRequest(path, "POST", json, new byte[0], null, FileUpload.class);
    }

    private ScanFileResponse scanUploadedFile(ScanFileRequest request, UUID fileID) {
        byte[] jsonBody;
        String path = "/v3/upload/" + fileID.toString() + "/scan";
        try {
            jsonBody = objectMapper.writeValueAsBytes((Object)request);
        }
        catch (JsonProcessingException e) {
            throw new NightfallClientException("processing scan file request: " + e.getMessage());
        }
        MediaType json = MediaType.parse((String)"application/json");
        return this.issueRequest(path, "POST", json, jsonBody, null, ScanFileResponse.class);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <E> E issueRequest(String path, String method, MediaType mediaType, byte[] body, Headers headers, Class<E> responseClass) {
        String url = this.apiHost + path;
        Request.Builder builder = new Request.Builder().url(url);
        if (headers != null) {
            builder.headers(headers);
        }
        if (this.implVersion != null && !this.implVersion.equals("")) {
            builder.addHeader("User-Agent", "nightfall-java-sdk/" + this.implVersion);
        }
        builder.addHeader("Authorization", "Bearer " + this.apiKey);
        RequestBody reqBody = null;
        if (body != null && body.length > 0) {
            reqBody = RequestBody.create((byte[])body, (MediaType)mediaType);
        } else if (!method.equals("GET") && !method.equals("HEAD")) {
            reqBody = RequestBody.create((byte[])new byte[0]);
        }
        builder.method(method, reqBody);
        Request request = builder.build();
        Call call = this.httpClient.newCall(request);
        NightfallErrorResponse lastError = null;
        int errorCode = 0;
        for (int attempt = 0; attempt < this.retryCount; ++attempt) {
            try (Response response = call.execute();){
                Object object;
                if (!response.isSuccessful()) {
                    try {
                        lastError = (NightfallErrorResponse)objectMapper.readValue(response.body().bytes(), NightfallErrorResponse.class);
                    }
                    catch (Throwable throwable) {
                        // empty catch block
                    }
                    if (response.code() == 429 && attempt < this.retryCount - 1) {
                        Thread.sleep(1000L);
                        call = call.clone();
                        continue;
                    }
                    errorCode = response.code();
                    break;
                }
                if (Void.class.equals(responseClass)) {
                    object = null;
                    return (E)object;
                }
                object = objectMapper.readValue(response.body().bytes(), responseClass);
                return (E)object;
            }
            catch (IOException e) {
                if (!e.getMessage().equalsIgnoreCase("timeout")) {
                    if (!e.getMessage().equalsIgnoreCase("read timed out")) throw new NightfallClientException("issuing HTTP request: " + e.getMessage());
                }
                if (attempt >= this.retryCount - 1) {
                    throw new NightfallRequestTimeoutException("request timed out");
                }
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                call = call.clone();
                continue;
            }
            catch (Throwable t) {
                throw new NightfallClientException("failure executing HTTP request: " + t.getMessage());
            }
        }
        if (errorCode > 0) {
            throw new NightfallAPIException("unsuccessful response", lastError, errorCode);
        }
        String message = "exceeded max retry count on request: " + path;
        throw new NightfallAPIException(message, lastError, 429);
    }

    public static class Builder {
        private String apiKey;
        private int fileUploadConcurrency = 1;
        private Duration connectionTimeout = Duration.ofSeconds(10L);
        private Duration readTimeout = Duration.ofSeconds(30L);
        private Duration writeTimeout = Duration.ofSeconds(60L);
        private int maxIdleConnections = 100;
        private Duration keepAliveDuration = Duration.ofSeconds(30L);
        private List<Interceptor> interceptors = new ArrayList<Interceptor>();

        public static NightfallClient defaultClient() {
            ConnectionPool cxnPool = new ConnectionPool(100, 30L, TimeUnit.SECONDS);
            OkHttpClient httpClient = new OkHttpClient.Builder().connectTimeout(Duration.ofSeconds(10L)).readTimeout(Duration.ofSeconds(30L)).writeTimeout(Duration.ofSeconds(60L)).connectionPool(cxnPool).build();
            return new NightfallClient(NightfallClient.API_HOST, Builder.readAPIKeyFromEnvironment(), 1, httpClient);
        }

        public Builder withAPIKey(String apiKey) {
            this.apiKey = apiKey;
            return this;
        }

        public Builder withFileUploadConcurrency(int concurrency) {
            if (concurrency <= 0 || concurrency > 100) {
                throw new IllegalArgumentException("fileUploadConcurrency must be in range [1,100]");
            }
            this.fileUploadConcurrency = concurrency;
            return this;
        }

        public Builder withConnectionTimeout(Duration connectionTimeout) {
            if (connectionTimeout == null || connectionTimeout.isNegative() || connectionTimeout.getSeconds() > 60L) {
                throw new IllegalArgumentException("connectionTimeout must be a non-negative duration <= 60 seconds");
            }
            this.connectionTimeout = connectionTimeout;
            return this;
        }

        public Builder withReadTimeout(Duration readTimeout) {
            if (readTimeout == null || readTimeout.isNegative() || readTimeout.getSeconds() > 120L) {
                throw new IllegalArgumentException("readTimeout must be a non-negative duration <= 120 seconds");
            }
            this.readTimeout = readTimeout;
            return this;
        }

        public Builder withWriteTimeout(Duration writeTimeout) {
            if (writeTimeout == null || writeTimeout.isNegative() || writeTimeout.getSeconds() > 120L) {
                throw new IllegalArgumentException("writeTimeout must be a non-negative duration <= 120 seconds");
            }
            this.writeTimeout = writeTimeout;
            return this;
        }

        public Builder withMaxIdleConnections(int maxIdleConnections) {
            if (maxIdleConnections < 1 || maxIdleConnections > 500) {
                throw new IllegalArgumentException("maxIdleConnections must be in the range [1, 500]");
            }
            this.maxIdleConnections = maxIdleConnections;
            return this;
        }

        public Builder withKeepAliveDuration(Duration keepAliveDuration) {
            if (keepAliveDuration == null || keepAliveDuration.isNegative() || keepAliveDuration.isZero() || keepAliveDuration.getSeconds() > 120L) {
                throw new IllegalArgumentException("keepAliveDuration must be a positive duration <= 120 seconds");
            }
            this.keepAliveDuration = keepAliveDuration;
            return this;
        }

        public Builder withInterceptor(Interceptor interceptor) {
            this.interceptors.add(interceptor);
            return this;
        }

        public NightfallClient build() {
            if (this.apiKey == null || this.apiKey.equals("")) {
                this.apiKey = Builder.readAPIKeyFromEnvironment();
            }
            ConnectionPool cxnPool = new ConnectionPool(this.maxIdleConnections, this.keepAliveDuration.toMillis(), TimeUnit.MILLISECONDS);
            OkHttpClient.Builder httpClientBuilder = new OkHttpClient.Builder().connectTimeout(this.connectionTimeout).readTimeout(this.readTimeout).writeTimeout(this.writeTimeout).connectionPool(cxnPool);
            for (Interceptor interceptor : this.interceptors) {
                httpClientBuilder = httpClientBuilder.addInterceptor(interceptor);
            }
            OkHttpClient httpClient = httpClientBuilder.build();
            return new NightfallClient(NightfallClient.API_HOST, this.apiKey, this.fileUploadConcurrency, httpClient);
        }

        private static String readAPIKeyFromEnvironment() {
            String apiKey = System.getenv("NIGHTFALL_API_KEY");
            if (apiKey == null || apiKey.equals("")) {
                throw new IllegalArgumentException("Missing value for NIGHTFALL_API_KEY environment variable");
            }
            return apiKey;
        }
    }
}

