/*
    Copyright 2023 Picovoice Inc.

    You may not use this file except in compliance with the license. A copy of the license is
    located in the "LICENSE" file accompanying this source.

    Unless required by applicable law or agreed to in writing, software distributed under the
    License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
    express or implied. See the License for the specific language governing permissions and
    limitations under the License.
*/

package ai.picovoice.android.voiceprocessor;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageManager;
import android.media.AudioFormat;
import android.media.AudioRecord;
import android.media.MediaRecorder;
import android.os.Handler;
import android.os.Looper;
import android.os.Process;

import androidx.core.content.ContextCompat;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * The Android Voice Processor is an asynchronous audio recorder designed for real-time
 * audio processing. Given some specifications, the library delivers frames of raw audio
 * data to the user via listeners. Audio will be 16-bit and mono.
 */
public class VoiceProcessor {

    private static VoiceProcessor instance = null;

    private final ArrayList<VoiceProcessorFrameListener> frameListeners = new ArrayList<>();
    private final ArrayList<VoiceProcessorErrorListener> errorListeners = new ArrayList<>();
    private final AtomicBoolean isStopRequested = new AtomicBoolean(false);
    private final Handler callbackHandler = new Handler(Looper.getMainLooper());
    private final Object listenerLock = new Object();

    private Future<Void> readThread = null;

    private int frameLength;
    private int sampleRate;

    private VoiceProcessor() {
    }

    /**
     * Obtain singleton instance of the VoiceProcessor.
     *
     * @return VoiceProcessor instance
     */
    public static synchronized VoiceProcessor getInstance() {
        if (instance == null) {
            instance = new VoiceProcessor();
        }

        return instance;
    }

    /**
     * Indicates whether the VoiceProcessor is currently recording or not.
     *
     * @return boolean indicating whether the VoiceProcessor is currently recording.
     */
    public boolean getIsRecording() {
        return readThread != null;
    }

    /**
     * Indicates whether the given context has been granted RECORD_AUDIO permissions or not.
     *
     * @return boolean indicating whether RECORD_AUDIO permission has been granted.
     */
    public boolean hasRecordAudioPermission(Context context) {
        return ContextCompat.checkSelfPermission(
                context,
                android.Manifest.permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
    }

    /**
     * Add a frame listener that will receive audio frames generated by the VoiceProcessor.
     *
     * @param listener VoiceProcessorFrameListener for processing frames of audio.
     */
    public void addFrameListener(VoiceProcessorFrameListener listener) {
        synchronized (listenerLock) {
            frameListeners.add(listener);
        }
    }

    /**
     * Add multiple frame listeners that will receive audio frames generated by the VoiceProcessor.
     *
     * @param listeners VoiceProcessorFrameListeners for processing frames of audio.
     */
    public void addFrameListeners(VoiceProcessorFrameListener[] listeners) {
        synchronized (listenerLock) {
            frameListeners.addAll(Arrays.asList(listeners));
        }
    }

    /**
     * Remove a frame listener from the VoiceProcessor. It will no longer receive audio frames.
     *
     * @param listener VoiceProcessorFrameListener that you would like to remove.
     */
    public void removeFrameListener(VoiceProcessorFrameListener listener) {
        synchronized (listenerLock) {
            frameListeners.remove(listener);
        }
    }

    /**
     * Remove frame listeners from the VoiceProcessor. They will no longer receive audio frames.
     *
     * @param listeners VoiceProcessorFrameListeners that you would like to remove.
     */
    public void removeFrameListeners(VoiceProcessorFrameListener[] listeners) {
        synchronized (listenerLock) {
            frameListeners.removeAll(Arrays.asList(listeners));
        }
    }

    /**
     * Clear all frame listeners from the VoiceProcessor. They will no longer receive audio frames.
     */
    public void clearFrameListeners() {
        synchronized (listenerLock) {
            frameListeners.clear();
        }
    }

    /**
     * Get number of frame listeners that are currently subscribed to the VoiceProcessor.
     *
     * @return the number of frame listeners
     */
    public int getNumFrameListeners() {
        return frameListeners.size();
    }

    /**
     * Add an error listener that will receive errors generated by the VoiceProcessor.
     *
     * @param errorListener VoiceProcessorErrorListener for catching recording errors.
     */
    public void addErrorListener(VoiceProcessorErrorListener errorListener) {
        synchronized (listenerLock) {
            errorListeners.add(errorListener);
        }
    }

    /**
     * Remove an error listener from the VoiceProcessor that had previously been added.
     *
     * @param errorListener VoiceProcessorErrorListener for catching recording errors.
     */
    public void removeErrorListener(VoiceProcessorErrorListener errorListener) {
        synchronized (listenerLock) {
            errorListeners.remove(errorListener);
        }
    }

    /**
     * Clear all error listeners from the VoiceProcessor.
     */
    public void clearErrorListeners() {
        synchronized (listenerLock) {
            errorListeners.clear();
        }
    }

    /**
     * Get number of error listeners that are currently subscribed to the VoiceProcessor.
     *
     * @return the number of error listeners
     */
    public int getNumErrorListeners() {
        return errorListeners.size();
    }

    /**
     * Starts audio capture. You need to subscribe a VoiceProcessorFrameListener via
     * {@link #addFrameListener(VoiceProcessorFrameListener)} in order to receive audio
     * frames from the VoiceProcessor.
     *
     * @param requestedFrameLength Number of audio samples per frame.
     * @param requestedSampleRate  Audio sample rate that the audio will be captured with.
     * @throws VoiceProcessorArgumentException if VoiceProcessor is already recording with
     *                                         a different configuration
     */
    public synchronized void start(
            final int requestedFrameLength,
            final int requestedSampleRate) throws VoiceProcessorArgumentException {
        if (getIsRecording()) {
            if (requestedFrameLength != frameLength || requestedSampleRate != sampleRate) {
                throw new VoiceProcessorArgumentException(
                        String.format(
                                "VoiceProcessor start() was called with frame length " +
                                        "%d and sample rate %d while already recording with " +
                                        "frame length %d and sample rate %d",
                                requestedFrameLength,
                                requestedSampleRate,
                                frameLength,
                                sampleRate));
            } else {
                return;
            }
        }

        frameLength = requestedFrameLength;
        sampleRate = requestedSampleRate;
        readThread = Executors.newSingleThreadExecutor().submit(new Callable<Void>() {
            @Override
            public Void call() {
                android.os.Process.setThreadPriority(Process.THREAD_PRIORITY_URGENT_AUDIO);
                read(frameLength, sampleRate);
                return null;
            }
        });
    }

    /**
     * Stops audio capture. Frames will stop being delivered to the subscribed listeners.
     *
     * @throws VoiceProcessorException if an error is encountered while trying to stop the
     *                                 recorder thread.
     */
    public synchronized void stop() throws VoiceProcessorException {
        if (!getIsRecording()) {
            return;
        }

        isStopRequested.set(true);
        try {
            readThread.get();
            readThread = null;
        } catch (ExecutionException | InterruptedException e) {
            throw new VoiceProcessorException(
                    "An error was encountered while requesting to stop the audio recording",
                    e);
        } finally {
            isStopRequested.set(false);
        }
    }

    @SuppressLint({"MissingPermission", "DefaultLocale"})
    private void read(int frameLength, int sampleRate) {
        final int minBufferSize = AudioRecord.getMinBufferSize(
                sampleRate,
                AudioFormat.CHANNEL_IN_MONO,
                AudioFormat.ENCODING_PCM_16BIT);
        final int bufferSize = Math.max(sampleRate / 2, minBufferSize);

        AudioRecord recorder;
        try {
            recorder = new AudioRecord(
                    MediaRecorder.AudioSource.MIC,
                    sampleRate,
                    AudioFormat.CHANNEL_IN_MONO,
                    AudioFormat.ENCODING_PCM_16BIT,
                    bufferSize);
        } catch (IllegalArgumentException e) {
            onError(new VoiceProcessorArgumentException(
                    "Unable to initialize audio recorder with required parameters",
                    e));
            return;
        }

        if (recorder.getState() != AudioRecord.STATE_INITIALIZED) {
            onError(new VoiceProcessorStateException(
                    "Audio recorder did not initialize successfully. " +
                            "Ensure you have acquired permission to record audio from the user."));
            return;
        }

        final short[] frame = new short[frameLength];
        try {
            recorder.startRecording();

            while (!isStopRequested.get()) {
                final int numSamplesRead = recorder.read(frame, 0, frame.length);
                if (numSamplesRead == frame.length) {
                    onFrame(frame);
                } else {
                    onError(new VoiceProcessorReadException(
                            String.format(
                                    "Expected a frame of size %d, but read one of size %d",
                                    frame.length,
                                    numSamplesRead)
                    ));
                }
            }

            recorder.stop();
        } catch (IllegalStateException e) {
            onError(new VoiceProcessorStateException(
                    "Audio recorder entered invalid state",
                    e));
        } finally {
            recorder.release();
        }
    }

    private void onFrame(final short[] frame) {
        synchronized (listenerLock) {
            for (final VoiceProcessorFrameListener listener : frameListeners) {
                callbackHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        listener.onFrame(frame);
                    }
                });
            }
        }
    }

    private void onError(final VoiceProcessorException e) {
        synchronized (listenerLock) {
            for (final VoiceProcessorErrorListener listener : errorListeners) {
                callbackHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        listener.onError(e);
                    }
                });
            }
        }
    }
}