/*
 * Decompiled with CFR 0.152.
 */
package ai.pipestream.api.registration;

import jakarta.enterprise.context.ApplicationScoped;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Base64;
import java.util.Optional;
import org.jboss.logging.Logger;

@ApplicationScoped
public class IconResourceLoader {
    private static final Logger LOG = Logger.getLogger(IconResourceLoader.class);
    private static final String SVG_DATA_URI_PREFIX = "data:image/svg+xml;base64,";
    private static final int MAX_ICON_SIZE_BYTES = 65536;
    private final HttpClient httpClient = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(10L)).build();

    public Optional<String> loadIcon(String iconPath) {
        if (iconPath == null || iconPath.trim().isEmpty()) {
            return Optional.empty();
        }
        String trimmedPath = iconPath.trim();
        try {
            if (trimmedPath.startsWith(SVG_DATA_URI_PREFIX)) {
                return this.validateAndReturnDataUri(trimmedPath);
            }
            if (trimmedPath.startsWith("classpath:")) {
                return this.loadFromClasspath(trimmedPath.substring("classpath:".length()));
            }
            if (trimmedPath.startsWith("http://") || trimmedPath.startsWith("https://")) {
                return this.loadFromUrl(trimmedPath);
            }
            return this.loadFromFileSystem(trimmedPath);
        }
        catch (Exception e) {
            LOG.warnf("Failed to load icon from path '%s': %s", (Object)iconPath, (Object)e.getMessage());
            return Optional.empty();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Optional<String> loadFromClasspath(String resourcePath) {
        String cleanPath = resourcePath.startsWith("/") ? resourcePath.substring(1) : resourcePath;
        try (InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(cleanPath);){
            if (inputStream == null) {
                LOG.warnf("Classpath resource not found: %s", (Object)cleanPath);
                Optional<String> optional2 = Optional.empty();
                return optional2;
            }
            byte[] svgBytes = inputStream.readAllBytes();
            Optional<String> optional = this.validateAndConvertToDataUri(svgBytes, "classpath:" + cleanPath);
            return optional;
        }
        catch (IOException e) {
            LOG.warnf("Error reading classpath resource '%s': %s", (Object)cleanPath, (Object)e.getMessage());
            return Optional.empty();
        }
    }

    private Optional<String> loadFromFileSystem(String filePath) {
        try {
            Path path = Paths.get(filePath, new String[0]);
            if (!Files.exists(path, new LinkOption[0])) {
                LOG.warnf("File not found: %s", (Object)filePath);
                return Optional.empty();
            }
            byte[] svgBytes = Files.readAllBytes(path);
            return this.validateAndConvertToDataUri(svgBytes, filePath);
        }
        catch (IOException e) {
            LOG.warnf("Error reading file '%s': %s", (Object)filePath, (Object)e.getMessage());
            return Optional.empty();
        }
    }

    private Optional<String> loadFromUrl(String url) {
        try {
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(url)).timeout(Duration.ofSeconds(30L)).GET().build();
            HttpResponse<byte[]> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofByteArray());
            if (response.statusCode() != 200) {
                LOG.warnf("HTTP error loading icon from URL '%s': %d", (Object)url, (Object)response.statusCode());
                return Optional.empty();
            }
            return this.validateAndConvertToDataUri(response.body(), url);
        }
        catch (Exception e) {
            LOG.warnf("Error loading icon from URL '%s': %s", (Object)url, (Object)e.getMessage());
            return Optional.empty();
        }
    }

    private Optional<String> validateAndConvertToDataUri(byte[] svgBytes, String source) {
        if (svgBytes.length == 0) {
            LOG.warnf("Empty SVG file from source: %s", (Object)source);
            return Optional.empty();
        }
        if (svgBytes.length > 65536) {
            LOG.warnf("SVG file too large (%d bytes, max %d) from source: %s", (Object)svgBytes.length, (Object)65536, (Object)source);
            return Optional.empty();
        }
        String svgContent = new String(svgBytes);
        if (!this.isValidSvg(svgContent)) {
            LOG.warnf("Invalid SVG content from source: %s", (Object)source);
            return Optional.empty();
        }
        String base64 = Base64.getEncoder().encodeToString(svgBytes);
        String dataUri = SVG_DATA_URI_PREFIX + base64;
        LOG.debugf("Successfully loaded SVG icon (%d bytes) from: %s", svgBytes.length, (Object)source);
        return Optional.of(dataUri);
    }

    private Optional<String> validateAndReturnDataUri(String dataUri) {
        try {
            String base64Part = dataUri.substring(SVG_DATA_URI_PREFIX.length());
            byte[] decodedBytes = Base64.getDecoder().decode(base64Part);
            if (decodedBytes.length > 65536) {
                LOG.warnf("Inline SVG data URI too large: %d bytes", (Object)decodedBytes.length);
                return Optional.empty();
            }
            String svgContent = new String(decodedBytes);
            if (!this.isValidSvg(svgContent)) {
                LOG.warn((Object)"Invalid SVG content in data URI");
                return Optional.empty();
            }
            return Optional.of(dataUri);
        }
        catch (IllegalArgumentException e) {
            LOG.warnf("Invalid base64 data in SVG data URI: %s", (Object)e.getMessage());
            return Optional.empty();
        }
    }

    private boolean isValidSvg(String content) {
        if (content == null || content.trim().isEmpty()) {
            return false;
        }
        String trimmed = content.trim();
        if (!trimmed.contains("<svg")) {
            return false;
        }
        if (!trimmed.startsWith("<") || !trimmed.endsWith(">")) {
            return false;
        }
        String lowerContent = trimmed.toLowerCase();
        if (lowerContent.contains("<script") || lowerContent.contains("javascript:") || lowerContent.contains("onclick") || lowerContent.contains("onload")) {
            LOG.warn((Object)"SVG contains potentially unsafe content");
            return false;
        }
        return true;
    }

    public String getDefaultServiceIcon() {
        String defaultSvg = "<svg width=\"24\" height=\"24\" viewBox=\"0 0 24 24\" fill=\"none\" xmlns=\"http://www.w3.org/2000/svg\">\n    <path d=\"M12 15.5A3.5 3.5 0 0 1 8.5 12A3.5 3.5 0 0 1 12 8.5a3.5 3.5 0 0 1 3.5 3.5a3.5 3.5 0 0 1-3.5 3.5m7.43-2.53c.04-.32.07-.64.07-.97c0-.33-.03-.66-.07-1l2.11-1.63c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.31-.61-.22l-2.49 1c-.52-.39-1.06-.73-1.69-.98l-.37-2.65A.506.506 0 0 0 14 2h-4c-.25 0-.46.18-.5.42l-.37 2.65c-.63.25-1.17.59-1.69.98l-2.49-1c-.22-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64L4.57 11c-.04.34-.07.67-.07 1c0 .33.03.65.07.97l-2.11 1.66c-.19.15-.25.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1.01c.52.4 1.06.74 1.69.99l.37 2.65c.04.24.25.42.5.42h4c.25 0 .46-.18.5-.42l.37-2.65c.63-.26 1.17-.59 1.69-.99l2.49 1.01c.22.08.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.66Z\" fill=\"currentColor\"/>\n</svg>\n";
        byte[] svgBytes = defaultSvg.getBytes();
        String base64 = Base64.getEncoder().encodeToString(svgBytes);
        return SVG_DATA_URI_PREFIX + base64;
    }
}

